<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  Vmpayment.Icard
 *
 * @copyright   Copyright (C) 2025 iCard AD. All rights reserved.
 * @license     GNU General Public License version 3 or later; see http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace Icard\Plugin\Vmpayment\Icard\UI;

defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;

/**
 * iCard Refund UI Renderer
 * Handles all refund-related HTML/JS generation
 * 
 * @package     Icard.Plugin
 * @subpackage  Vmpayment.Icard.UI
 * @since       1.0.0
 */
class RefundUIRenderer
{
    private $repository;

    public function __construct($repository)
    {
        $this->repository = $repository;
    }

    /**
     * Render the refund panel in backend order details
     *
     * @param   int     $virtuemart_order_id   Order ID
     * @param   object  $paymentTable          Payment table data
     * @param   int     $payment_method_id     Payment method ID
     * @param   object  $order                 Order data
     * @param   object  $method                Payment method configuration
     *
     * @return  string  HTML output
     */
    public function renderRefundPanel($virtuemart_order_id, $paymentTable, $payment_method_id, $order, $method)
    {
        $orderStatus = $order['details']['BT']->order_status;
        $orderTotal = $paymentTable->payment_order_total;
        $currency = $paymentTable->payment_currency;
        $transactionId = $paymentTable->icard_transaction_id ?? '';
        $currencyCode = $this->getCurrencyCode($currency);
        
        // Get payment error information
        $paymentError = null;
        if ($orderStatus === 'X' && !empty($paymentTable->icard_response_message)) {
            $paymentError = [
                'message' => $paymentTable->icard_response_message,
                'code' => $paymentTable->icard_response_code ?? '',
                'status' => $paymentTable->icard_status ?? 'failed'
            ];
        }
        
        // Calculate refund amounts
        $totalRefunded = $paymentTable->icard_total_refunded ?? 0;
        $remainingAmount = $orderTotal - $totalRefunded;
        
        // Check refund eligibility
        $canRefund = (
            ($orderStatus === 'C' || $orderStatus === 'R') && 
            !empty($transactionId) && 
            $remainingAmount > 0 &&
            $method->allow_refunds == 1
        );
        
        $isFullyRefunded = ($orderStatus === 'R' && $totalRefunded >= $orderTotal);
        $isPartiallyRefunded = (($orderStatus === 'R' || $orderStatus === 'C') && $totalRefunded > 0 && $totalRefunded < $orderTotal);
        
        return $this->buildRefundPanelHTML(
            $virtuemart_order_id,
            $orderStatus,
            $orderTotal,
            $currencyCode,
            $transactionId,
            $totalRefunded,
            $remainingAmount,
            $canRefund,
            $isFullyRefunded,
            $isPartiallyRefunded,
            $method,
            $paymentError
        );
    }

    /**
     * Build refund panel HTML
     */
    private function buildRefundPanelHTML(
        $orderId,
        $orderStatus,
        $orderTotal,
        $currencyCode,
        $transactionId,
        $totalRefunded,
        $remainingAmount,
        $canRefund,
        $isFullyRefunded,
        $isPartiallyRefunded,
        $method = null,
        $paymentError = null
    ) {
        $html = '<div class="icard-refund-panel" style="margin-top: 15px; margin-bottom: 15px; padding: 12px; border: 1px solid #ddd; border-radius: 4px; background: #f8f9fa; width: 100%;">';
        
        // Header
        $html .= $this->renderHeader();
        
        // Info row
        $html .= $this->renderInfoRow($transactionId, $orderTotal, $totalRefunded, $currencyCode, $orderStatus);
        
        // Show payment error if exists
        if ($paymentError) {
            $html .= $this->renderPaymentError($paymentError);
        }
        
        // Action section
        // Don't show refund warnings for cancelled orders - it's redundant
        if ($orderStatus === 'X') {
            // Order is cancelled, no need to show refund options or warnings
        } elseif ($method && $method->allow_refunds != 1) {
            $html .= $this->renderRefundsDisabledSection();
        } elseif ($isFullyRefunded) {
            $html .= $this->renderFullyRefundedSection($totalRefunded, $currencyCode);
        } elseif ($isPartiallyRefunded) {
            $html .= $this->renderPartiallyRefundedSection($totalRefunded, $remainingAmount, $currencyCode, $orderId);
        } elseif ($canRefund) {
            $html .= $this->renderRefundInputSection($totalRefunded, $remainingAmount, $currencyCode, $orderId);
        } else {
            $html .= $this->renderCannotRefundSection($transactionId, $orderStatus, $remainingAmount, $totalRefunded, $currencyCode, $method);
        }
        
        $html .= '</div>';
        
        // Add JavaScript
        $html .= $this->getRefundJavaScript();
        
        return $html;
    }

    /**
     * Render panel header
     */
    private function renderHeader()
    {
        $html = '<div style="display: flex; align-items: center; margin-bottom: 10px; border-bottom: 1px solid #e9ecef; padding-bottom: 8px;">';
        $html .= '<span style="font-size: 16px; margin-right: 8px;">💳</span>';
        $html .= '<strong style="color: #048EF8; font-size: 14px;">iCard Refund</strong>';
        $html .= '</div>';
        
        return $html;
    }

    /**
     * Render info row
     */
    private function renderInfoRow($transactionId, $orderTotal, $totalRefunded, $currencyCode, $orderStatus)
    {
        $html = '<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; font-size: 12px;">';
        $html .= '<span><strong>ID:</strong> ' . htmlspecialchars(substr($transactionId, -8) ?: 'N/A') . '</span>';
        $html .= '<span><strong>Total:</strong> ' . number_format($orderTotal, 2) . ' ' . htmlspecialchars($currencyCode);
        if ($totalRefunded > 0) {
            $html .= ' <span style="color: #dc3545;">(Refunded: ' . number_format($totalRefunded, 2) . ')</span>';
        }
        $html .= '</span>';
        $html .= '<span class="icard-status-badge">' . $this->getOrderStatusBadge($orderStatus) . '</span>';
        $html .= '</div>';
        
        return $html;
    }

    /**
     * Render payment error message with beautiful styling
     */
    private function renderPaymentError($paymentError)
    {
        $html = '<div style="padding: 14px 16px; background: #fce4ec; border: 1px solid #f8bbd0; border-radius: 6px; margin-bottom: 12px; box-shadow: 0 1px 3px rgba(211, 47, 47, 0.1);">';
        $html .= '<div style="display: flex; align-items: flex-start; gap: 10px;">';
        $html .= '<span style="font-size: 20px; flex-shrink: 0; margin-top: 1px;">❌</span>';
        $html .= '<div style="flex: 1;">';
        $html .= '<strong style="display: block; margin-bottom: 6px; color: #c62828; font-size: 14px;">Payment Failed</strong>';
        $html .= '<div style="color: #b71c1c; font-size: 13px; line-height: 1.5;">' . htmlspecialchars($paymentError['message']) . '</div>';
        if (!empty($paymentError['code'])) {
            $html .= '<div style="margin-top: 6px; font-size: 11px; color: #d32f2f; opacity: 0.85; font-family: monospace; background: rgba(255,255,255,0.5); padding: 3px 6px; border-radius: 3px; display: inline-block;">Error Code: ' . htmlspecialchars($paymentError['code']) . '</div>';
        }
        $html .= '</div>';
        $html .= '</div>';
        $html .= '</div>';
        
        return $html;
    }

    /**
     * Render fully refunded section
     */
    private function renderFullyRefundedSection($totalRefunded, $currencyCode)
    {
        $html = '<div style="padding: 8px; background: #d4edda; border: 1px solid #c3e6cb; border-radius: 3px; color: #155724; font-size: 12px;">';
        $html .= '<strong>✓ Fully Refunded</strong><br>';
        $html .= 'Amount: ' . number_format($totalRefunded, 2) . ' ' . htmlspecialchars($currencyCode);
        $html .= '</div>';
        
        return $html;
    }

    /**
     * Render partially refunded section (with refund input)
     */
    private function renderPartiallyRefundedSection($totalRefunded, $remainingAmount, $currencyCode, $orderId)
    {
        $html = '<div style="padding: 8px; background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 3px; color: #856404; font-size: 12px; margin-bottom: 10px;">';
        $html .= '<strong>⚠️ Partial Refund Applied</strong><br>';
        $html .= 'Refunded: ' . number_format($totalRefunded, 2) . ' ' . htmlspecialchars($currencyCode) . '<br>';
        $html .= 'Remaining: ' . number_format($remainingAmount, 2) . ' ' . htmlspecialchars($currencyCode);
        $html .= '</div>';
        
        // Show refund input for remaining amount
        $html .= $this->renderRefundInput($remainingAmount, $currencyCode, $orderId);
        
        return $html;
    }

    /**
     * Render refund input section (for confirmed orders)
     */
    private function renderRefundInputSection($totalRefunded, $remainingAmount, $currencyCode, $orderId)
    {
        $html = '<div style="text-align: center;">';
        
        // Show previous refunds info if any
        if ($totalRefunded > 0) {
            $html .= '<div style="margin-bottom: 10px; padding: 6px; background: #d1ecf1; border: 1px solid #bee5eb; border-radius: 3px; color: #0c5460; font-size: 11px;">';
            $html .= '<strong>Previous refunds:</strong> ' . number_format($totalRefunded, 2) . ' ' . htmlspecialchars($currencyCode);
            $html .= '</div>';
        }
        
        $html .= $this->renderRefundInput($remainingAmount, $currencyCode, $orderId);
        $html .= '</div>';
        
        return $html;
    }

    /**
     * Render refund input and button
     */
    private function renderRefundInput($remainingAmount, $currencyCode, $orderId)
    {
        $html = '<div style="margin-bottom: 10px; padding: 8px; background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 3px;">';
        $html .= '<label for="icard_refund_amount_' . $orderId . '" style="font-size: 12px; font-weight: bold; display: block; margin-bottom: 5px;">Refund Amount (Remaining: ' . number_format($remainingAmount, 2) . ' ' . htmlspecialchars($currencyCode) . '):</label>';
        $html .= '<div style="display: flex; align-items: center; justify-content: center; gap: 5px;">';
        $html .= '<input type="number" id="icard_refund_amount_' . $orderId . '" name="icard_refund_amount" step="0.01" min="0.01" max="' . number_format($remainingAmount, 2, '.', '') . '" value="' . number_format($remainingAmount, 2, '.', '') . '" style="width: 100px; padding: 4px; border: 1px solid #ccc; border-radius: 3px; font-size: 12px; text-align: right;">';
        $html .= '<span style="font-size: 12px; font-weight: bold;">' . htmlspecialchars($currencyCode) . '</span>';
        $html .= '</div>';
        $html .= '</div>';
        
        // Process refund button
        $html .= '<div style="display: flex; gap: 10px; justify-content: center;">';
        $html .= '<button type="button" class="btn btn-primary btn-sm" onclick="icardProcessRefund(' . $orderId . ', ' . number_format($remainingAmount, 2, '.', '') . ')" style="padding: 8px 16px; font-size: 13px; font-weight: bold; background: #007bff; border: none; border-radius: 4px; color: white;">';
        $html .= '💰 Process Refund';
        $html .= '</button>';
        $html .= '</div>';
        
        $html .= '<p style="margin: 5px 0 0 0; font-size: 11px; color: #666;">Refund amount will determine status: Full amount = Refunded, Partial = Confirmed</p>';
        
        return $html;
    }

    /**
     * Render refunds disabled section
     */
    private function renderRefundsDisabledSection()
    {
        $html = '<div style="padding: 12px; background: #f8d7da; border: 2px solid #f5c6cb; border-radius: 4px; color: #721c24; font-size: 14px; text-align: center; margin: 10px 0;">';
        $html .= '<div style="display: flex; align-items: center; justify-content: center; gap: 8px; margin-bottom: 8px;">';
        $html .= '<span style="font-size: 20px;">🚫</span>';
        $html .= '<strong style="font-size: 16px;">Refunds Disabled</strong>';
        $html .= '</div>';
        $html .= '<p style="margin: 0; font-weight: 500;">Refunds are currently disabled for this payment method.</p>';
        $html .= '</div>';
        
        return $html;
    }

    /**
     * Render cannot refund section
     */
    private function renderCannotRefundSection($transactionId, $orderStatus, $remainingAmount, $totalRefunded, $currencyCode, $method = null)
    {
        $html = '<div style="padding: 8px; background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 3px; color: #856404; font-size: 12px;">';
        $html .= '<strong>⚠️ Cannot Refund</strong>';
        
        // Check if refunds are disabled
        if ($method && $method->allow_refunds != 1) {
            $html .= '<br>Refunds are disabled for this payment method';
        } elseif (empty($transactionId)) {
            $html .= '<br>No transaction ID';
        } elseif ($orderStatus !== 'C' && $orderStatus !== 'R') {
            $html .= '<br>Status must be Confirmed or Refunded';
        } elseif ($remainingAmount <= 0) {
            $html .= '<br>Order fully refunded (Total: ' . number_format($totalRefunded, 2) . ' ' . htmlspecialchars($currencyCode) . ')';
        }
        $html .= '</div>';
        
        return $html;
    }

    /**
     * Render transaction details table
     *
     * @param   int     $virtuemart_order_id  Order ID
     * @param   object  $paymentTable         Payment table data
     *
     * @return  string  HTML output
     */
    public function renderTransactionTable($virtuemart_order_id, $paymentTable)
    {
        $transactions = $this->repository->getTransactionHistory($virtuemart_order_id, $paymentTable);
        
        $html = '<table class="adminlist table">' . "\n";
        $html .= '<thead><tr style="display: flex;"><th colspan="2" style="padding: 10px; text-align: center; font-weight: bold;">💳 iCard Checkout Transaction Details</th></tr></thead>' . "\n";
        $html .= '<tbody>' . "\n";
        
        foreach ($transactions as $transaction) {
            $html .= '<tr>';
            $html .= '<td style="font-weight: bold; width: 120px;">' . $transaction['type'] . ' ID:</td>';
            $html .= '<td>' . htmlspecialchars($transaction['transaction_id']) . '</td>';
            $html .= '<td style="font-weight: bold; width: 100px;">Date:</td>';
            $html .= '<td>' . htmlspecialchars($transaction['date']) . '</td>';
            $html .= '<td style="font-weight: bold; width: 80px;">Amount:</td>';
            $html .= '<td style="text-align: right; font-weight: bold; color: ' . $transaction['color'] . ';">' . $transaction['amount'] . '</td>';
            $html .= '</tr>' . "\n";
        }
        
        $html .= '</tbody>' . "\n";
        $html .= '</table>' . "\n";
        
        return $html;
    }

    /**
     * Get order status badge HTML with beautiful styling
     */
    private function getOrderStatusBadge($status)
    {
        $badges = [
            'P' => ['label' => 'Pending', 'color' => '#f57c00', 'bg' => '#fff3e0', 'border' => '#ffb74d'],
            'C' => ['label' => 'Confirmed', 'color' => '#2e7d32', 'bg' => '#e8f5e9', 'border' => '#81c784'],
            'X' => ['label' => 'Cancelled', 'color' => '#d32f2f', 'bg' => '#fce4ec', 'border' => '#f8bbd0'],
            'R' => ['label' => 'Refunded', 'color' => '#5e35b1', 'bg' => '#ede7f6', 'border' => '#b39ddb'],
            'S' => ['label' => 'Shipped', 'color' => '#0277bd', 'bg' => '#e1f5fe', 'border' => '#81d4fa'],
        ];
        
        $badge = $badges[$status] ?? ['label' => $status, 'color' => '#616161', 'bg' => '#f5f5f5', 'border' => '#e0e0e0'];
        
        return '<span style="display: inline-block; padding: 6px 14px; border-radius: 6px; background: ' . $badge['bg'] . '; color: ' . $badge['color'] . '; font-weight: 600; font-size: 12px; border: 1px solid ' . $badge['border'] . '; box-shadow: 0 1px 2px rgba(0,0,0,0.05);">' 
            . htmlspecialchars($badge['label']) . '</span>';
    }

    /**
     * Get JavaScript for refund functionality
     */
    private function getRefundJavaScript()
    {
        ob_start();
        ?>
    <script type='text/javascript'>
    // VirtueMart-style toast notification system
    function showToast(message, type = 'info', duration = 5000) {
        // Remove existing toasts
        var existingToasts = document.querySelectorAll('.icard-toast');
        existingToasts.forEach(function(toast) {
            toast.remove();
        });
        
        // Create toast container if it doesn't exist
        var toastContainer = document.getElementById('icard-toast-container');
        if (!toastContainer) {
            toastContainer = document.createElement('div');
            toastContainer.id = 'icard-toast-container';
            toastContainer.style.cssText = 'position: fixed; top: 20px; right: 20px; z-index: 9999; max-width: 400px;';
            document.body.appendChild(toastContainer);
        }
        
        // Create toast element
        var toast = document.createElement('div');
        toast.className = 'icard-toast';
        
        // Set colors based on type
        var colors = {
            'success': { bg: '#d4edda', border: '#c3e6cb', color: '#155724', icon: '✅' },
            'error': { bg: '#f8d7da', border: '#f5c6cb', color: '#721c24', icon: '❌' },
            'warning': { bg: '#fff3cd', border: '#ffeaa7', color: '#856404', icon: '⚠️' },
            'info': { bg: '#d1ecf1', border: '#bee5eb', color: '#0c5460', icon: 'ℹ️' }
        };
        
        var toastStyle = colors[type] || colors['info'];
        
        toast.style.cssText = `
            background: ${toastStyle.bg};
            border: 1px solid ${toastStyle.border};
            color: ${toastStyle.color};
            padding: 15px 20px;
            margin-bottom: 10px;
            border-radius: 4px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            font-size: 14px;
            line-height: 1.4;
            position: relative;
            opacity: 0;
            transform: translateX(100%);
            transition: all 0.3s ease;
            max-width: 100%;
            word-wrap: break-word;
        `;
        
        toast.innerHTML = `
            <div style="display: flex; align-items: flex-start; gap: 10px;">
                <span style="font-size: 16px; flex-shrink: 0; margin-top: 2px;">${toastStyle.icon}</span>
                <div style="flex: 1;">
                    <div style="font-weight: 600; margin-bottom: 5px;">${type.charAt(0).toUpperCase() + type.slice(1)}</div>
                    <div>${message}</div>
                </div>
                <button onclick="this.parentElement.parentElement.remove()" style="
                    background: none;
                    border: none;
                    color: ${toastStyle.color};
                    font-size: 18px;
                    cursor: pointer;
                    padding: 0;
                    margin-left: 10px;
                    opacity: 0.7;
                    flex-shrink: 0;
                ">&times;</button>
            </div>
        `;
        
        toastContainer.appendChild(toast);
        
        // Animate in
        setTimeout(function() {
            toast.style.opacity = '1';
            toast.style.transform = 'translateX(0)';
        }, 100);
        
        // Auto remove
        setTimeout(function() {
            if (toast.parentElement) {
                toast.style.opacity = '0';
                toast.style.transform = 'translateX(100%)';
                setTimeout(function() {
                    if (toast.parentElement) {
                        toast.remove();
                    }
                }, 300);
            }
        }, duration);
    }
    
    // VirtueMart-style confirmation modal
    function showConfirmModal(title, message, onConfirm, onCancel) {
        // Remove existing modal
        var existingModal = document.getElementById('icard-confirm-modal');
        if (existingModal) {
            existingModal.remove();
        }
        
        // Create modal overlay
        var modal = document.createElement('div');
        modal.id = 'icard-confirm-modal';
        modal.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 10000;
            display: flex;
            align-items: center;
            justify-content: center;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        `;
        
        // Create modal content
        var modalContent = document.createElement('div');
        modalContent.style.cssText = `
            background: white;
            border-radius: 8px;
            padding: 30px;
            max-width: 500px;
            width: 90%;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            text-align: center;
        `;
        
        modalContent.innerHTML = `
            <div style="font-size: 24px; margin-bottom: 15px;">⚠️</div>
            <h3 style="margin: 0 0 15px 0; color: #333; font-size: 18px;">${title}</h3>
            <p style="margin: 0 0 25px 0; color: #666; line-height: 1.5; font-size: 14px;">${message}</p>
            <div style="display: flex; gap: 15px; justify-content: center;">
                <button id="icard-confirm-cancel" style="
                    background: #6c757d;
                    color: white;
                    border: none;
                    padding: 10px 20px;
                    border-radius: 4px;
                    cursor: pointer;
                    font-size: 14px;
                    font-weight: 500;
                ">Cancel</button>
                <button id="icard-confirm-ok" style="
                    background: #ffc107;
                    color: #212529;
                    border: none;
                    padding: 10px 20px;
                    border-radius: 4px;
                    cursor: pointer;
                    font-size: 14px;
                    font-weight: 500;
                    transition: all 0.2s ease;
                " onmouseover="this.style.background='#e0a800'" onmouseout="this.style.background='#ffc107'">Confirm Refund</button>
            </div>
        `;
        
        modal.appendChild(modalContent);
        document.body.appendChild(modal);
        
        // Add event listeners
        document.getElementById('icard-confirm-ok').onclick = function() {
            modal.remove();
            if (onConfirm) onConfirm();
        };
        
        document.getElementById('icard-confirm-cancel').onclick = function() {
            modal.remove();
            if (onCancel) onCancel();
        };
        
        modal.onclick = function(e) {
            if (e.target === modal) {
                modal.remove();
                if (onCancel) onCancel();
            }
        };
    }
    
    function icardProcessRefund(orderId, maxAmount) {
        var refundAmount = document.getElementById('icard_refund_amount_' + orderId);
        var amount = refundAmount ? parseFloat(refundAmount.value) : 0;
        
        if (amount <= 0) {
            showToast('Please enter a valid refund amount', 'error');
            return false;
        }
        
        if (amount > maxAmount) {
            showToast('Refund amount cannot exceed ' + maxAmount, 'error');
            return false;
        }
        
        // Determine if this is a full or partial refund
        var isFullRefund = Math.abs(amount - maxAmount) < 0.01;
        var newStatus = isFullRefund ? 'R' : 'C';
        var statusText = isFullRefund ? 'Refunded' : 'Confirmed';
        
        // Show confirmation modal instead of browser confirm
        showConfirmModal(
            'Confirm Refund Processing',
            'Are you sure you want to process this refund?<br><br><strong>Amount:</strong> ' + amount + '<br><br><strong>This will:</strong><br>• Process refund via iCard gateway<br>• Change order status to ' + statusText + '<br>• Cannot be undone',
            function() {
                // Find the original "Process Refund" button (the blue one with 💰 icon)
                var refundButton = document.querySelector('button[onclick*="icardProcessRefund(' + orderId + '"]');
                if (!refundButton) {
                    // Fallback: find button containing "Process Refund" text
                    var buttons = document.querySelectorAll('button');
                    for (var i = 0; i < buttons.length; i++) {
                        if (buttons[i].innerHTML.includes('Process Refund') && buttons[i].innerHTML.includes('💰')) {
                            refundButton = buttons[i];
                            break;
                        }
                    }
                }
                
                if (refundButton) {
                    // Store original content
                    var originalText = refundButton.innerHTML;
                    var originalStyle = refundButton.style.cssText;
                    
                    // Create loading animation
                    refundButton.innerHTML = `
                        <div style="display: flex; align-items: center; justify-content: center; gap: 8px;">
                            <div class="icard-spinner" style="
                                width: 16px;
                                height: 16px;
                                border: 2px solid #ffffff;
                                border-top: 2px solid transparent;
                                border-radius: 50%;
                                animation: icard-spin 1s linear infinite;
                            "></div>
                            <span>Processing Refund...</span>
                        </div>
                    `;
                    
                    // Add spinner animation keyframes if not exists
                    if (!document.getElementById('icard-spinner-styles')) {
                        var style = document.createElement('style');
                        style.id = 'icard-spinner-styles';
                        style.textContent = `
                            @keyframes icard-spin {
                                0% { transform: rotate(0deg); }
                                100% { transform: rotate(360deg); }
                            }
                        `;
                        document.head.appendChild(style);
                    }
                    
                    // Disable button and change appearance
                    refundButton.disabled = true;
                    refundButton.style.background = '#6c757d';
                    refundButton.style.cursor = 'not-allowed';
                    refundButton.style.opacity = '0.8';
                    
                    // Store original values for restoration
                    refundButton.dataset.originalText = originalText;
                    refundButton.dataset.originalStyle = originalStyle;
                    
                    // Process refund directly via AJAX
                    icardProcessRefundDirect(orderId, amount, newStatus, statusText, refundButton);
                }
            },
            function() {
                // User cancelled
            }
        );
        
        return true;
    }

    // Helper to get Joomla CSRF token field name
    function icardGetCSRFTokenName() {
        // Try meta option
        if (window.Joomla && window.Joomla.getOptions) {
            var name = window.Joomla.getOptions('csrf.token');
            if (name) return name;
        }
        // Try hidden inputs in forms
        var tokenInput = document.querySelector('input[name*="token"]');
        if (tokenInput) return tokenInput.name;
        return 'token';
    }

    function icardProcessRefundDirect(orderId, amount, newStatus, statusText, button) {
        var formData = new FormData();
        formData.append('order_id', orderId);
        formData.append('amount', amount);
        formData.append('new_status', newStatus);
        formData.append('status_text', statusText);
        var csrfName = icardGetCSRFTokenName();
        formData.append(csrfName, 1);
        
        fetch('index.php?option=com_ajax&plugin=icard&group=vmpayment&format=json&task=process_refund_direct', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Success state
                button.innerHTML = `
                    <div style="display: flex; align-items: center; justify-content: center; gap: 8px;">
                        <span style="font-size: 16px;">✅</span>
                        <span>${statusText}</span>
                    </div>
                `;
                button.style.background = '#28a745';
                button.style.cursor = 'default';
                
                updateOrderStatusDisplay(statusText, newStatus);
                
                var successMessage = 'Refund processed successfully!<br><br><strong>Amount:</strong> ' + amount + '<br><strong>Status:</strong> ' + statusText;
                if (data.transaction_id && data.transaction_id !== 'N/A') {
                    successMessage += '<br><strong>Transaction ID:</strong> ' + data.transaction_id;
                }
                
                showToast(successMessage, 'success', 6000);
                
                setTimeout(function() {
                    window.location.reload();
                }, 4000);
            } else {
                // Error state
                button.innerHTML = `
                    <div style="display: flex; align-items: center; justify-content: center; gap: 8px;">
                        <span style="font-size: 16px;">❌</span>
                        <span>Failed</span>
                    </div>
                `;
                button.style.background = '#dc3545';
                button.style.cursor = 'not-allowed';
                
                showToast('Refund failed: ' + (data.message || 'Unknown error'), 'error', 8000);
                
                // Restore button after delay
                setTimeout(function() {
                    restoreButton(button);
                }, 5000);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            
            // Network error state
            button.innerHTML = `
                <div style="display: flex; align-items: center; justify-content: center; gap: 8px;">
                    <span style="font-size: 16px;">❌</span>
                    <span>Network Error</span>
                </div>
            `;
            button.style.background = '#dc3545';
            button.style.cursor = 'not-allowed';
            
            showToast('Network error processing refund. Please try again.', 'error', 8000);
            
            // Restore button after delay
            setTimeout(function() {
                restoreButton(button);
            }, 5000);
        });
    }
    
    // Helper function to restore button to original state
    function restoreButton(button) {
        if (button.dataset.originalText && button.dataset.originalStyle) {
            button.innerHTML = button.dataset.originalText;
            button.style.cssText = button.dataset.originalStyle;
            button.disabled = false;
            button.removeAttribute('data-original-text');
            button.removeAttribute('data-original-style');
        }
    }

function updateOrderStatusDisplay(statusText, newStatus) {
    var statusBadge = document.querySelector('.icard-status-badge');
    if (statusBadge) {
        var color = newStatus === 'R' ? '#6c757d' : '#ffc107';
        var bgColor = newStatus === 'R' ? '#e2e3e5' : '#fff3cd';
        statusBadge.innerHTML = statusText;
        statusBadge.style.color = color;
        statusBadge.style.background = bgColor;
    }
    
    var statusSelect = document.getElementById('order_status') || 
                     document.querySelector('select[name="order_status"]') ||
                     document.querySelector('select[name="jform[order_status]"]');
    if (statusSelect) {
        statusSelect.value = newStatus;
    }
}
</script>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get currency code
     */
    private function getCurrencyCode($currencyId)
    {
        try {
            $db = Factory::getContainer()->get('DatabaseDriver');
            $query = $db->getQuery(true)
                ->select('currency_code_3')
                ->from('#__virtuemart_currencies')
                ->where($db->quoteName('virtuemart_currency_id') . ' = ' . (int)$currencyId);
            $db->setQuery($query);
            return $db->loadResult() ?: 'EUR';
        } catch (\Exception $e) {
            return 'EUR';
        }
    }
}

