/**
 * iCard Toast Notification Service
 * Centralized notification system for the iCard plugin
 */
(function() {
    'use strict';
    
    // Toast Service Singleton
    window.IcardToastService = {
        // Configuration
        config: {
            position: 'top-right',
            duration: 5000,
            maxToasts: 5,
            zIndex: 9999
        },
        
        // Toast types with styling
        types: {
            'success': { 
                bg: '#d4edda', 
                border: '#c3e6cb', 
                color: '#155724', 
                icon: '✅',
                title: 'Success'
            },
            'error': { 
                bg: '#f8d7da', 
                border: '#f5c6cb', 
                color: '#721c24', 
                icon: '❌',
                title: 'Error'
            },
            'warning': { 
                bg: '#fff3cd', 
                border: '#ffeaa7', 
                color: '#856404', 
                icon: '⚠️',
                title: 'Warning'
            },
            'info': { 
                bg: '#d1ecf1', 
                border: '#bee5eb', 
                color: '#0c5460', 
                icon: 'ℹ️',
                title: 'Information'
            },
            'loading': { 
                bg: '#e2e3e5', 
                border: '#d6d8db', 
                color: '#383d41', 
                icon: '🔄',
                title: 'Loading'
            }
        },
        
        // Active toasts counter
        activeToasts: 0,
        
        /**
         * Show a toast notification
         * @param {string} message - The message to display
         * @param {string} type - Toast type (success, error, warning, info, loading)
         * @param {object} options - Additional options
         */
        show: function(message, type = 'info', options = {}) {
            // Merge options with defaults
            var settings = Object.assign({}, this.config, options);
            
            // Validate type
            if (!this.types[type]) {
                type = 'info';
            }
            
            // Remove oldest toasts if we exceed max
            this._cleanupOldToasts(settings.maxToasts);
            
            // Create toast container if it doesn't exist
            this._ensureContainer();
            
            // Create and show toast
            var toast = this._createToast(message, type, settings);
            this._showToast(toast);
            
            // Auto-remove after duration (except loading)
            if (type !== 'loading' && settings.duration > 0) {
                setTimeout(() => {
                    this._removeToast(toast);
                }, settings.duration);
            }
            
            return toast;
        },
        
        /**
         * Show success toast
         */
        success: function(message, options = {}) {
            return this.show(message, 'success', options);
        },
        
        /**
         * Show error toast
         */
        error: function(message, options = {}) {
            return this.show(message, 'error', options);
        },
        
        /**
         * Show warning toast
         */
        warning: function(message, options = {}) {
            return this.show(message, 'warning', options);
        },
        
        /**
         * Show info toast
         */
        info: function(message, options = {}) {
            return this.show(message, 'info', options);
        },
        
        /**
         * Show loading toast
         */
        loading: function(message, options = {}) {
            return this.show(message, 'loading', options);
        },
        
        /**
         * Remove all toasts
         */
        clear: function() {
            var container = document.getElementById('icard-toast-container');
            if (container) {
                container.innerHTML = '';
                this.activeToasts = 0;
            }
        },
        
        /**
         * Remove specific toast
         */
        remove: function(toast) {
            this._removeToast(toast);
        },
        
        /**
         * Update toast message (useful for loading toasts)
         */
        update: function(toast, message, type = null) {
            if (!toast || !toast.querySelector) return;
            
            var messageEl = toast.querySelector('.icard-toast-message');
            if (messageEl) {
                messageEl.textContent = message;
            }
            
            if (type && this.types[type]) {
                var iconEl = toast.querySelector('.icard-toast-icon');
                if (iconEl) {
                    iconEl.textContent = this.types[type].icon;
                }
                
                // Update styling
                var toastStyle = this.types[type];
                toast.style.background = toastStyle.bg;
                toast.style.borderColor = toastStyle.border;
                toast.style.color = toastStyle.color;
            }
        },
        
        /**
         * Ensure toast container exists
         */
        _ensureContainer: function() {
            var container = document.getElementById('icard-toast-container');
            if (!container) {
                container = document.createElement('div');
                container.id = 'icard-toast-container';
                container.style.cssText = `
                    position: fixed;
                    top: 20px;
                    right: 20px;
                    z-index: ${this.config.zIndex};
                    max-width: 400px;
                    pointer-events: none;
                `;
                document.body.appendChild(container);
            }
        },
        
        /**
         * Create toast element
         */
        _createToast: function(message, type, settings) {
            var toast = document.createElement('div');
            toast.className = 'icard-toast';
            toast.setAttribute('data-type', type);
            toast.setAttribute('data-timestamp', Date.now());
            
            var toastStyle = this.types[type];
            
            // Base styling
            toast.style.cssText = `
                background: ${toastStyle.bg};
                border: 1px solid ${toastStyle.border};
                color: ${toastStyle.color};
                padding: 20px 25px;
                margin-bottom: 10px;
                border-radius: 4px;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                font-size: 14px;
                line-height: 1.4;
                position: relative;
                opacity: 0;
                transform: translateX(100%);
                transition: all 0.3s ease;
                max-width: 100%;
                word-wrap: break-word;
                pointer-events: auto;
                cursor: pointer;
            `;
            
            // Add close button
            var closeBtn = document.createElement('button');
            closeBtn.innerHTML = '×';
            closeBtn.style.cssText = `
                position: absolute;
                top: 5px;
                right: 8px;
                background: none;
                border: none;
                font-size: 18px;
                font-weight: bold;
                color: inherit;
                cursor: pointer;
                opacity: 0.7;
                padding: 0;
                width: 20px;
                height: 20px;
                display: flex;
                align-items: center;
                justify-content: center;
            `;
            
            // Toast content
            var content = document.createElement('div');
            content.style.cssText = 'display: flex; align-items: flex-start; gap: 10px;';
            
            var icon = document.createElement('span');
            icon.className = 'icard-toast-icon';
            icon.textContent = toastStyle.icon;
            icon.style.cssText = 'font-size: 16px; flex-shrink: 0; margin-top: 2px;';
            
            var messageEl = document.createElement('div');
            messageEl.className = 'icard-toast-message';
            messageEl.textContent = message;
            messageEl.style.cssText = 'flex: 1; min-width: 0; line-height: 27px;';
            
            content.appendChild(icon);
            content.appendChild(messageEl);
            
            toast.appendChild(content);
            toast.appendChild(closeBtn);
            
            // Add click to close
            toast.addEventListener('click', () => {
                this._removeToast(toast);
            });
            
            closeBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this._removeToast(toast);
            });
            
            return toast;
        },
        
        /**
         * Show toast with animation
         */
        _showToast: function(toast) {
            var container = document.getElementById('icard-toast-container');
            if (!container) return;
            
            container.appendChild(toast);
            this.activeToasts++;
            
            // Trigger animation
            requestAnimationFrame(() => {
                toast.style.opacity = '1';
                toast.style.transform = 'translateX(0)';
            });
        },
        
        /**
         * Remove toast with animation
         */
        _removeToast: function(toast) {
            if (!toast || !toast.parentNode) return;
            
            toast.style.opacity = '0';
            toast.style.transform = 'translateX(100%)';
            
            setTimeout(() => {
                if (toast.parentNode) {
                    toast.parentNode.removeChild(toast);
                    this.activeToasts--;
                }
            }, 300);
        },
        
        /**
         * Clean up old toasts
         */
        _cleanupOldToasts: function(maxToasts) {
            var container = document.getElementById('icard-toast-container');
            if (!container) return;
            
            var toasts = container.querySelectorAll('.icard-toast');
            if (toasts.length >= maxToasts) {
                // Remove oldest toasts
                var toastsArray = Array.from(toasts);
                toastsArray.sort((a, b) => {
                    return parseInt(a.getAttribute('data-timestamp')) - parseInt(b.getAttribute('data-timestamp'));
                });
                
                var toRemove = toastsArray.slice(0, toastsArray.length - maxToasts + 1);
                toRemove.forEach(toast => this._removeToast(toast));
            }
        }
    };
    
    // Auto-initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            // Service is ready
        });
    }
    
})();
