<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  Vmpayment.Icard
 *
 * @copyright   Copyright (C) 2025 iCard AD. All rights reserved.
 * @license     GNU General Public License version 3 or later; see http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace Icard\Plugin\Vmpayment\Icard\Service;

defined('_JEXEC') or die('Restricted access');

/**
 * iCard Signature Service
 * Centralized RSA signature creation and verification
 * 
 * @package     Icard.Plugin
 * @subpackage  Vmpayment.Icard.Service
 * @since       1.0.0
 */
class SignatureService
{
    private $logger;

    public function __construct($logger = null)
    {
        $this->logger = $logger;
    }

    /**
     * Create RSA signature for iCard API
     *
     * @param   array   $data        Data array to sign
     * @param   string  $privateKey  Private key for signing
     *
     * @return  string|false  Base64 encoded signature or false on error
     */
    public function createSignature($data, $privateKey)
    {
        if (empty($privateKey)) {
            $this->log('Signature Service: Signature creation failed: Empty private key', 'ERROR');
            return false;
        }

        // Log the input data (hide signature if present)
        $logData = $data;
        if (isset($logData['Signature'])) {
            unset($logData['Signature']);
        }
        $this->log('Signature Service: Input data: ' . json_encode($logData, JSON_PRETTY_PRINT), 'INFO');

        // Build canonical signature string
        $signatureData = $this->buildSignatureData($data);
        $this->log('Signature Service: Canonical signature string: ' . $signatureData, 'INFO');
        
        // Format private key for OpenSSL
        $privateKeyPem = $this->formatPrivateKey($privateKey);
        if (!$privateKeyPem) {
            $this->log('Signature Service: Signature creation failed: Invalid private key format', 'ERROR');
            return false;
        }

        $pkey = openssl_pkey_get_private($privateKeyPem);
        if ($pkey === false) {
            $this->log('Signature Service: Signature creation failed: OpenSSL could not load private key', 'ERROR');
            return false;
        }

        $signature = '';
        if (!openssl_sign($signatureData, $signature, $pkey, OPENSSL_ALGO_SHA256)) {
            $this->log('Signature Service: Signature creation failed: OpenSSL sign operation failed', 'ERROR');
            return false;
        }

        $base64Signature = base64_encode($signature);
        
        return $base64Signature;
    }

    /**
     * Verify RSA signature for iCard notifications
     *
     * @param   array   $data       Data array to verify
     * @param   string  $publicKey  Public key for verification
     * @param   string  $signature  Base64 encoded signature
     *
     * @return  bool  True if signature is valid
     */
    public function verifySignature($data, $publicKey, $signature)
    {
        if (empty($publicKey) || empty($signature)) {
            return false;
        }
        
        $publicKeyResource = openssl_pkey_get_public($publicKey);
        if ($publicKeyResource === false) {
            return false;
        }

        // Prepare data for signature verification (remove signature fields)
        $verifyData = $data;
        unset($verifyData['signature']);
        unset($verifyData['Signature']);

        // Build signature data string (same format as when creating signature)
        $signatureData = $this->buildSignatureData($verifyData);
        
        // Decode and verify signature
        $decodedSignature = base64_decode($signature);

        if ($decodedSignature === false) {
            return false;
        }

        // Verify signature with SHA256
        $result = openssl_verify($signatureData, $decodedSignature, $publicKeyResource, OPENSSL_ALGO_SHA256);
        
        if ($result === 1) {
            return true;
        }
        
        return false;
    }

    /**
     * Build canonical signature string for consistent signing
     *
     * @param   array  $data  Data array
     *
     * @return  string  Canonical signature string
     */
    private function buildSignatureData($data)
    {
        // Remove signature fields if present
        if (isset($data['Signature'])) {
            unset($data['Signature']);
        }
        if (isset($data['signature'])) {
            unset($data['signature']);
        }

        // Flatten nested arrays using colon delimiter (like WooCommerce)
        $data = $this->flattenNested('', $data, ':');
        
        // Filter out empty arrays
        $data = array_filter($data, function ($value) {
            if (is_array($value) && empty($value)) {
                return false;
            }
            return true;
        });

        // Convert keys to lowercase
        $data = array_change_key_case($data);

        // Build key:value pairs
        $pairs = [];
        foreach ($data as $key => $value) {
            if (is_bool($value)) {
                $value = intval($value);
            }

            if (is_string($value)) {
                $value = mb_convert_encoding($value, 'UTF8');
            }

            if ($value === null || $value === '') {
                $pairs[] = $key . ':';
            } else {
                $pairs[] = $key . ':' . $value;
            }
        }

        sort($pairs, SORT_NATURAL);
        return implode(';', $pairs);
    }

    /**
     * Flatten nested arrays for signature generation
     * Matches PrestaShop implementation exactly
     *
     * @param   string  $prefix     Prefix for keys
     * @param   array   $array      Array to flatten
     * @param   string  $delimiter  Key delimiter
     *
     * @return  array  Flattened array
     */
    private function flattenNested($prefix, $array, $delimiter = '.')
    {
        $data = [];
        foreach ($array as $key => $value) {
            if ($prefix) {
                $key = $prefix . $delimiter . $key;
            }
            if (is_array($value)) {
                $data = array_merge($data, $this->flattenNested($key, $value, $delimiter));
            } else {
                $data[$key] = $value;
            }
        }
        return $data;
    }

    /**
     * Format private key for OpenSSL
     *
     * @param   string  $key  Private key string
     *
     * @return  string|false  Formatted private key or false on error
     */
    private function formatPrivateKey($key)
    {
        if (empty($key)) return false;

        // Add headers if missing
        if (strpos($key, '-----BEGIN') === false) {
            $key = "-----BEGIN RSA PRIVATE KEY-----\n" . trim($key) . "\n-----END RSA PRIVATE KEY-----";
        }

        // Normalize line endings
        $key = str_replace(["\r\n", "\r"], "\n", $key);

        // Ensure proper line lengths for base64 body
        $lines = explode("\n", $key);
        if (count($lines) === 1) {
            $key = chunk_split($key, 64, "\n");
        }

        return $key;
    }

    /**
     * Log message using provided logger callback
     *
     * @param   string  $message  Message to log
     * @param   string  $level    Log level
     *
     * @return  void
     */
    private function log($message, $level = 'INFO')
    {
        if ($this->logger && is_callable($this->logger)) {
            call_user_func($this->logger, $message, $level);
        }
    }
}
