<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  Vmpayment.Icard
 *
 * @copyright   Copyright (C) 2025 iCard AD. All rights reserved.
 * @license     GNU General Public License version 3 or later; see http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace Icard\Plugin\Vmpayment\Icard\Service;

defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Uri\Uri;
use Icard\Plugin\Vmpayment\Icard\Service\SignatureService;

/**
 * iCard Refund Service
 * Handles all refund business logic
 * 
 * @package     Icard.Plugin
 * @subpackage  Vmpayment.Icard.Service
 * @since       1.0.0
 */
class RefundService
{
    private $repository;
    private $signatureService;
    private $logger;

    public function __construct($repository, $logger = null, $signatureService = null)
    {
        $this->repository = $repository;
        $this->logger = $logger;
        $this->signatureService = $signatureService ?? new SignatureService($logger);
    }

    /**
     * Process a refund request
     *
     * @param   object  $order          Order data
     * @param   object  $paymentTable   Payment table data
     * @param   object  $method         Payment method configuration
     * @param   float   $refundAmount   Amount to refund (null = full refund)
     * @param   string  $oldStatus      Previous order status
     *
     * @return  array   Result ['success' => bool, 'message' => string, 'data' => array]
     */
    public function processRefund($order, $paymentTable, $method, $refundAmount = null, $oldStatus = null)
    {
        try {
            $virtuemart_order_id = $order['details']['BT']->virtuemart_order_id;
            
            // Validate transaction exists
            if (empty($paymentTable->icard_transaction_id)) {
                return $this->errorResponse('No transaction ID found for this order');
            }
            
            // Validate refunds are enabled
            if (!$method->allow_refunds) {
                return $this->errorResponse('Refunds not enabled for this payment method');
            }
            
            // Calculate refund amounts
            $amounts = $this->calculateRefundAmounts($paymentTable, $refundAmount);
            
            // Validate refund eligibility
            $validation = $this->validateRefund($order, $paymentTable, $method, $amounts, $oldStatus);
            if (!$validation['valid']) {
                return $this->errorResponse($validation['message']);
            }
            
            // Call iCard API
            $apiResult = $this->callIcardRefundAPI($order, $paymentTable, $amounts['refund_amount'], $method);
            
            if (!$apiResult['success']) {
                return $this->errorResponse('Refund API call failed: ' . $apiResult['error']);
            }
            
            // Store refund data using repository
            $this->repository->storeRefundData(
                $virtuemart_order_id,
                $apiResult,
                $amounts['refund_amount'],
                $paymentTable
            );
            
            // Determine new status
            $newStatus = $this->determineOrderStatus($amounts);
            
            // Refund processed successfully
            
            return [
                'success' => true,
                'message' => 'Refund processed successfully',
                'data' => [
                    'transaction_id' => $apiResult['transaction_id'],
                    'refund_amount' => $amounts['refund_amount'],
                    'new_status' => $newStatus,
                    'is_full_refund' => $amounts['is_full_refund']
                ]
            ];
            
        } catch (\Exception $e) {
            return $this->errorResponse('Exception: ' . $e->getMessage());
        }
    }

    /**
     * Calculate refund amounts and determine refund type
     */
    private function calculateRefundAmounts($paymentTable, $requestedAmount = null)
    {
        $totalRefunded = $paymentTable->icard_total_refunded ?? 0;
        $orderTotal = $paymentTable->payment_order_total;
        $remainingAmount = $orderTotal - $totalRefunded;
        
        // Use requested amount or remaining amount
        $refundAmount = $requestedAmount ?? $remainingAmount;
        
        // Ensure refund amount doesn't exceed remaining amount
        if ($refundAmount > $remainingAmount) {
            $refundAmount = $remainingAmount;
        }
        
        // Determine if full refund (allow small rounding differences)
        $isFullRefund = abs($remainingAmount - $refundAmount) < 0.01;
        
        return [
            'order_total' => $orderTotal,
            'total_refunded' => $totalRefunded,
            'remaining_amount' => $remainingAmount,
            'refund_amount' => $refundAmount,
            'is_full_refund' => $isFullRefund,
            'remaining_after_refund' => $remainingAmount - $refundAmount
        ];
    }

    /**
     * Validate if refund can be processed
     */
    private function validateRefund($order, $paymentTable, $method, $amounts, $oldStatus = null)
    {
        // Check order status (if oldStatus provided, use it for validation)
        $statusToCheck = $oldStatus ?? $order['details']['BT']->order_status;
        
        // Allow refunds from Confirmed (C) or Refunded (R) status
        // Both statuses allow refunds as long as there's remaining amount
        if ($statusToCheck !== 'C' && $statusToCheck !== 'R') {
            return [
                'valid' => false,
                'message' => 'Order must be Confirmed (C) or Refunded (R) status to process refunds (current status: ' . $statusToCheck . ')'
            ];
        }
        
        // Check remaining amount
        if ($amounts['remaining_amount'] <= 0) {
            return [
                'valid' => false,
                'message' => 'Order fully refunded, no remaining amount'
            ];
        }
        
        // Validate refund amount
        if ($amounts['refund_amount'] <= 0) {
            return [
                'valid' => false,
                'message' => 'Invalid refund amount: ' . $amounts['refund_amount']
            ];
        }
        
        if ($amounts['refund_amount'] > $amounts['remaining_amount']) {
            return [
                'valid' => false,
                'message' => 'Refund amount (' . $amounts['refund_amount'] . ') exceeds remaining amount (' . $amounts['remaining_amount'] . ')'
            ];
        }
        
        return ['valid' => true];
    }

    /**
     * Call iCard Refund API
     */
    private function callIcardRefundAPI($order, $paymentTable, $refundAmount, $method)
    {
        try {
            $credentials = $this->getCredentials($method);
            $customerEmail = $this->getCustomerEmail($order);
            
            // Build refund request
            $refundData = [
                'IPGmethod' => 'IPGRefund',
                'IPGVersion' => '4.5',
                'KeyIndex' => $credentials['key_index'],
                'KeyIndexResp' => $credentials['key_response_index'],
                'Originator' => $credentials['client_id'],
                'OutputFormat' => 'json',
                'OrderID' => 'REFUND_' . $paymentTable->virtuemart_order_id . '_' . time(),
                'IPG_Trnref' => $paymentTable->icard_transaction_id,
                'MID' => $credentials['merchant_id'],
                'Currency' => $credentials['currency'],
                'Amount' => number_format($refundAmount, 2, '.', ''),
                'Email' => $customerEmail,
                'URL_Notify' => Uri::root() . 'index.php?option=com_virtuemart&view=pluginresponse&task=pluginnotification&pm=' . 
                    $paymentTable->virtuemart_paymentmethod_id
            ];
            
            // Create signature using centralized service
            $refundData['Signature'] = $this->signatureService->createSignature($refundData, $credentials['private_key']);
            
            if (!$refundData['Signature']) {
                $this->log_to_file('Refund API: Failed to create signature', 'ERROR');
                return ['success' => false, 'error' => 'Failed to create signature'];
            }
            
            // Make API call and verify response signature
            $result = $this->makeApiRequest($credentials['url'], $refundData, $credentials['public_key']);
            
            return $result;
            
        } catch (\Exception $e) {
            $this->log_to_file('Refund API: Exception: ' . $e->getMessage(), 'ERROR');
            return ['success' => false, 'error' => 'Exception: ' . $e->getMessage()];
        }
    }

    /**
     * Get credentials based on test mode
     */
    private function getCredentials($method)
    {
        $test = $method->test;
        $prefix = $test ? 'developer_' : 'production_';
        
        return [
            'merchant_id' => $method->{$prefix . 'merchant_id'},
            'client_id' => $method->{$prefix . 'client_id'},
            'currency' => $method->{$prefix . 'currency'},
            'private_key' => $method->{$prefix . 'private_key'},
            'public_key' => $method->{$prefix . 'public_key'},
            'key_index' => $method->{$prefix . 'key_index'},
            'key_response_index' => $method->{$prefix . 'key_response_index'},
            'url' => $method->{$prefix . 'url'}
        ];
    }

    /**
     * Get customer email from order
     */
    private function getCustomerEmail($order)
    {
        // Use the order data that was already loaded
        return ($order && isset($order['details']['BT'])) ? $order['details']['BT']->email : '';
    }


    /**
     * Make HTTP API request
     */
    private function makeApiRequest($url, $data, $publicKey)
    {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($data),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/x-www-form-urlencoded',
                'User-Agent: Joomla-VirtueMart-iCard/' . JVERSION
            ]
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        if ($curlError) {
            $this->log_to_file('Refund API: cURL error: ' . $curlError, 'ERROR');
            return ['success' => false, 'error' => 'cURL error: ' . $curlError];
        }
        
        if ($httpCode !== 200) {
            $this->log_to_file('Refund API: HTTP error ' . $httpCode, 'ERROR');
            return ['success' => false, 'error' => 'HTTP error: ' . $httpCode];
        }
        
        $responseData = json_decode($response, true);
        
        if (!$responseData) {
            $this->log_to_file('Refund API: Invalid JSON response', 'ERROR');
            return ['success' => false, 'error' => 'Invalid JSON response'];
        }
        
        // Log response (hide signature)
        $logResponse = $responseData;
        if (isset($logResponse['Signature'])) {
            $logResponse['Signature'] = '[HIDDEN]';
        }
        
        // SECURITY: Verify signature on response
        $receivedSignature = $responseData['Signature'] ?? $responseData['signature'] ?? null;
        
        if ($receivedSignature) {
            $isValid = $this->signatureService->verifySignature($responseData, $publicKey, $receivedSignature);
            
            if ($isValid) {
                $this->log_to_file('Refund API: Response signature verified successfully ✓', 'INFO');
            } else {
                $this->log_to_file('Refund API: Response signature verification FAILED ✗', 'ERROR');
                return ['success' => false, 'error' => 'Invalid signature on refund response'];
            }
        } else {
            $this->log_to_file('Refund API: No signature found in response', 'ERROR');
            return ['success' => false, 'error' => 'No signature in refund response'];
        }
        
        if (isset($responseData['Status']) && $responseData['Status'] == 0) {
            $this->log_to_file('Refund API: Refund successful', 'INFO');
            return [
                'success' => true,
                'refund_id' => $responseData['IPGTrnref'] ?? $responseData['IPC_Trnref'] ?? '',
                'transaction_id' => $responseData['IPGTrnref'] ?? $responseData['IPC_Trnref'] ?? '',
                'amount' => $responseData['Amount'] ?? 0,
                'currency' => $responseData['Currency'] ?? '',
                'message' => $responseData['Message'] ?? 'Refund successful'
            ];
        } else {
            $errorMsg = $responseData['Message'] ?? 'Refund failed';
            $this->log_to_file('Refund API: Refund failed - ' . $errorMsg, 'ERROR');
            return [
                'success' => false,
                'error' => $errorMsg,
                'code' => $responseData['Status'] ?? 'Unknown'
            ];
        }
    }

    /**
     * Determine order status based on refund type
     */
    private function determineOrderStatus($amounts)
    {
        if ($amounts['is_full_refund']) {
            return 'R';
        }
        
        return 'C';
    }

    /**
     * Log message using provided logger callback
     *
     * @param   string  $message  Message to log
     * @param   string  $level    Log level
     *
     * @return  void
     */
    private function log_to_file($message, $level = 'INFO')
    {
        if ($this->logger && is_callable($this->logger)) {
            call_user_func($this->logger, $message, $level);
        }
    }

    /**
     * Create error response
     */
    private function errorResponse($message)
    {
        return [
            'success' => false,
            'message' => $message,
            'data' => null
        ];
    }
}

