<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  Vmpayment.Icard
 *
 * @copyright   Copyright (C) 2025 iCard AD. All rights reserved.
 * @license     GNU General Public License version 3 or later; see http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace Icard\Plugin\Vmpayment\Icard\Service;

defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Icard\Plugin\Vmpayment\Icard\Service\SignatureService;

/**
 * Gateway Test Service for iCard Payment Plugin
 * Handles all gateway connection testing functionality
 * 
 * @package     Icard.Plugin
 * @subpackage  Vmpayment.Icard.Service
 * @since       1.0.0
 */
class GatewayTestService
{
    /**
     * Signature service
     *
     * @var SignatureService
     */
    private $signatureService;

    /**
     * Plugin instance for logging
     *
     * @var \Icard\Plugin\Vmpayment\Icard\Extension\Icard
     */
    private $plugin;

    /**
     * Constructor
     */
    public function __construct($plugin = null)
    {
        $this->plugin = $plugin;
        $this->signatureService = new SignatureService([$this, 'log_to_file']);
    }

    /**
     * Log message using VirtueMart's standard logging
     *
     * @param   string  $message  Message to log
     * @param   string  $level    Log level
     *
     * @return  void
     */
    private function log_to_file($message, $level = 'INFO')
    {
        if ($this->plugin) {
            $this->plugin->log_to_file($message, $level);
        }
    }

    /**
     * Handle test connection AJAX request
     *
     * @return  array  Result array
     */
    public function handleTestConnection()
    {
        $app = Factory::getApplication();
        $test_mode = $app->input->getInt('test_mode', 1);
        $method = $this->getPaymentMethod();
        
        if (!$method) {
            $this->log_to_file('iCard payment method not found', 'ERROR');
            return ['success' => false, 'message' => 'iCard payment method not found'];
        }
        
        $credentials = $this->getCredentials($method, $test_mode);
        $validation = $this->validateCredentials($credentials);
        
        if (!$validation['valid']) {
            $this->log_to_file('Credential validation failed: ' . $validation['message'], 'ERROR');
            return ['success' => false, 'message' => $validation['message']];
        }
        
        return $this->testGatewayConnection($credentials);
    }
    
    /**
     * Get payment method configuration
     *
     * @return  object|null  Payment method object
     */
    private function getPaymentMethod()
    {
        if (!$this->plugin) {
            return null;
        }

        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true)
            ->select('virtuemart_paymentmethod_id')
            ->from('#__virtuemart_paymentmethods')
            ->where($db->quoteName('payment_element') . ' = ' . $db->quote('icard'));
        
        $db->setQuery($query);
        $method_id = $db->loadResult();
        
        if (!$method_id) {
            return null;
        }

        return $this->plugin->getVmPluginMethod($method_id);
    }
    
    /**
     * Get credentials based on test mode
     *
     * @param   object  $method      Payment method object
     * @param   int     $test_mode   Test mode (1 = developer, 0 = production)
     *
     * @return  array  Credentials array
     */
    private function getCredentials($method, $test_mode)
    {
        $prefix = $test_mode == 1 ? 'developer_' : 'production_';
        $default_urls = [
            'developer' => 'https://dev-ipg2.icards.eu/trunk/',
            'production' => 'https://ipg.icard.com/'
        ];
        
        return [
            'merchant_id' => $method->{$prefix . 'merchant_id'},
            'client_id' => $method->{$prefix . 'client_id'},
            'private_key' => $method->{$prefix . 'private_key'},
            'public_key' => $method->{$prefix . 'public_key'},
            'key_index' => $method->{$prefix . 'key_index'},
            'key_response_index' => $method->{$prefix . 'key_response_index'},
            'url' => $method->{$prefix . 'url'} ?? $default_urls[$test_mode == 1 ? 'developer' : 'production'],
            'mode_name' => $test_mode == 1 ? 'Development' : 'Production'
        ];
    }
    
    /**
     * Validate required credentials
     *
     * @param   array  $credentials  Credentials array
     *
     * @return  array  Validation result
     */
    private function validateCredentials($credentials)
    {
        $required_fields = [
            'merchant_id' => 'Merchant ID',
            'client_id' => 'Client ID',
            'private_key' => 'Private Key',
            'public_key' => 'Public Key',
            'url' => 'Gateway URL',
            'key_index' => 'Key Index',
            'key_response_index' => 'Key Response Index'
        ];
        
        $missing = [];
        foreach ($required_fields as $field => $label) {
            if (empty($credentials[$field])) {
                $missing[] = $label;
            }
        }
        
        if (!empty($missing)) {
            return [
                'valid' => false,
                'message' => 'Missing required fields: ' . implode(', ', $missing)
            ];
        }

        return ['valid' => true];
    }
    
    /**
     * Test connection to iCard gateway
     *
     * @param   array  $credentials  Credentials array
     *
     * @return  array  Test result
     */
    private function testGatewayConnection($credentials)
    {
        $test_data = $this->buildTestData($credentials);
        $signature = $this->signatureService->createSignature($test_data, $credentials['private_key']);
        
        if ($signature === false) {
            $this->log_to_file('Failed to generate signature for ' . $credentials['mode_name'] . ' mode', 'ERROR');
            return [
                'success' => false,
                'message' => $credentials['mode_name'] . ' mode: Failed to generate signature. Check your private key format.'
            ];
        }
        
        $test_data['Signature'] = $signature;
        return $this->makeGatewayRequest($credentials['url'], $test_data, $credentials['mode_name']);
    }
    
    /**
     * Build test data for gateway
     *
     * @param   array  $credentials  Credentials array
     *
     * @return  array  Test data array
     */
    private function buildTestData($credentials)
    {
        return [
            'IPGmethod' => 'IPGSignatureVerify',
            'IPGVersion' => '4.5',
            'MID' => $credentials['merchant_id'],
            'KeyIndexResp' => $credentials['key_response_index'],
            'Originator' => $credentials['client_id'],
            'KeyIndex' => $credentials['key_index'],
            'Source' => 'joomla_virtuemart_' . JVERSION . '_php_' . PHP_VERSION,
            'URL_Notify' => Uri::root() . 'index.php?option=com_virtuemart&view=pluginresponse&task=pluginnotification',
            'OutputFormat' => 'json',
        ];
    }
    
    /**
     * Make HTTP request to gateway
     *
     * @param   string  $url        Gateway URL
     * @param   array   $data       Request data
     * @param   string  $mode_name  Mode name for error messages
     *
     * @return  array  Response result
     */
    private function makeGatewayRequest($url, $data, $mode_name)
    {
        try {
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => http_build_query($data),
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/x-www-form-urlencoded',
                    'User-Agent: Joomla-VirtueMart-iCard/' . JVERSION
                ]
            ]);
            
            // SSL settings - always verify SSL for security
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
            
            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curl_error = curl_error($ch);
            curl_close($ch);
            
            if ($curl_error) {
                return [
                    'success' => false,
                    'message' => $mode_name . ' mode: Connection error - ' . $curl_error
                ];
            }
            
            if ($http_code !== 200) {
                return [
                    'success' => false,
                    'message' => $mode_name . ' mode: HTTP error ' . $http_code . ' - ' . substr($response, 0, 100)
                ];
            }

            // Format JSON response for better readability
            $formatted_response = $this->formatJsonResponse($response);
            
            return $this->parseGatewayResponse($response, $mode_name);
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $mode_name . ' mode: Exception - ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Format JSON response for better logging
     *
     * @param   string  $response  Raw JSON response
     *
     * @return  string  Formatted response
     */
    private function formatJsonResponse($response)
    {
        $decoded = json_decode($response, true);
        if ($decoded) {
            // Remove signature from logging for security
            if (isset($decoded['Signature'])) {
                $decoded['Signature'] = '[HIDDEN]';
            }
            return json_encode($decoded, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        }
        return $response;
    }

    /**
     * Parse and validate gateway response
     *
     * @param   string  $response    Response string
     * @param   string  $mode_name   Mode name for error messages
     *
     * @return  array  Parsed result
     */
    private function parseGatewayResponse($response, $mode_name)
    {
        $result = json_decode($response, true);
        
        if (!$result) {
            return [
                'success' => false,
                'message' => $mode_name . ' mode: Invalid JSON response. Server returned: ' . substr($response, 0, 100)
            ];
        }
        
        $is_success = isset($result['status']) && $result['status'] == 0;
        
        if ($is_success) {
            $this->log_to_file('Connection successful! Gateway verified your credentials.', 'INFO');
            return [
                'success' => true,
                'message' => 'Connection successful! Gateway verified your credentials.'
            ];
        } else {
            $message = $result['status_msg'] ?? 'Gateway returned an error status';
            $this->log_to_file('Gateway test failed: ' . $message, 'ERROR');
            return [
                'success' => false,
                'message' => $message
            ];
        }
    }

}
