<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  Vmpayment.Icard
 *
 * @copyright   Copyright (C) 2025 iCard AD. All rights reserved.
 * @license     GNU General Public License version 3 or later; see http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace Icard\Plugin\Vmpayment\Icard\Repository;

defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;

/**
 * iCard Refund Repository
 * Handles all refund database operations
 * 
 * @package     Icard.Plugin
 * @subpackage  Vmpayment.Icard.Repository
 * @since       1.0.0
 */
class RefundRepository
{
    private $tableName;

    public function __construct($tableName)
    {
        $this->tableName = $tableName;
    }

    /**
     * Store refund data (accumulate, not replace)
     *
     * @param   int     $virtuemart_order_id  Order ID
     * @param   array   $refundResult         API response data
     * @param   float   $refundAmount         Refund amount
     * @param   object  $paymentTable         Existing payment table data
     *
     * @return  bool    Success status
     */
    public function storeRefundData($virtuemart_order_id, $refundResult, $refundAmount, $paymentTable)
    {
        try {
            $db = Factory::getContainer()->get('DatabaseDriver');
            
            // Get existing refund data
            $existingData = $this->getExistingRefundData($paymentTable);
            
            // Create unique refund ID
            $uniqueRefundId = 'REFUND_' . $virtuemart_order_id . '_' . time();
            $currentDate = date('Y-m-d H:i:s');
            
            // Accumulate refund data
            $updatedData = $this->accumulateRefundData($existingData, $uniqueRefundId, $refundAmount, $currentDate);
            
            // Update database
            $query = $db->getQuery(true);
            $query->update($this->tableName)
                  ->set($db->quoteName('icard_refund_transaction_id') . ' = ' . $db->quote($updatedData['transaction_ids']))
                  ->set($db->quoteName('icard_refund_amount') . ' = ' . $db->quote($refundAmount))
                  ->set($db->quoteName('icard_refund_date') . ' = NOW()')
                  ->set($db->quoteName('icard_refund_status') . ' = ' . $db->quote('success'))
                  ->set($db->quoteName('icard_refund_reason') . ' = ' . $db->quote('Admin refund'))
                  ->set($db->quoteName('icard_total_refunded') . ' = ' . $db->quote($updatedData['total_refunded']))
                  ->set($db->quoteName('icard_refund_count') . ' = ' . (int)$updatedData['refund_count'])
                  ->set($db->quoteName('icard_refund_amounts') . ' = ' . $db->quote($updatedData['amounts']))
                  ->set($db->quoteName('icard_refund_dates') . ' = ' . $db->quote($updatedData['dates']))
                  ->where($db->quoteName('virtuemart_order_id') . ' = ' . (int)$virtuemart_order_id);
            
            $db->setQuery($query);
            $db->execute();
            
            // Refund data stored successfully
            
            return true;
            
        } catch (\Exception $e) {
            // Error storing refund data - silent fail
            return false;
        }
    }

    /**
     * Get existing refund data from payment table
     */
    private function getExistingRefundData($paymentTable)
    {
        return [
            'transaction_ids' => $paymentTable->icard_refund_transaction_id ?? '',
            'amounts' => $paymentTable->icard_refund_amounts ?? '',
            'dates' => $paymentTable->icard_refund_dates ?? '',
            'total_refunded' => $paymentTable->icard_total_refunded ?? 0,
            'refund_count' => $paymentTable->icard_refund_count ?? 0
        ];
    }

    /**
     * Accumulate refund data with new refund
     */
    private function accumulateRefundData($existing, $newTransactionId, $newAmount, $newDate)
    {
        $result = [];
        
        // Accumulate transaction IDs
        if (!empty($existing['transaction_ids'])) {
            $ids = explode(',', $existing['transaction_ids']);
            $ids[] = $newTransactionId;
            $result['transaction_ids'] = implode(',', $ids);
        } else {
            $result['transaction_ids'] = $newTransactionId;
        }
        
        // Accumulate amounts
        if (!empty($existing['amounts'])) {
            $amounts = explode(',', $existing['amounts']);
            $amounts[] = $newAmount;
            $result['amounts'] = implode(',', $amounts);
        } else {
            $result['amounts'] = $newAmount;
        }
        
        // Accumulate dates
        if (!empty($existing['dates'])) {
            $dates = explode(',', $existing['dates']);
            $dates[] = $newDate;
            $result['dates'] = implode(',', $dates);
        } else {
            $result['dates'] = $newDate;
        }
        
        // Calculate totals
        $result['total_refunded'] = $existing['total_refunded'] + $newAmount;
        $result['refund_count'] = $existing['refund_count'] + 1;
        
        return $result;
    }

    /**
     * Update order status directly in database
     *
     * @param   int     $orderId     Order ID
     * @param   string  $newStatus   New status code
     * @param   string  $statusText  Status text for history
     * @param   float   $amount      Refund amount
     *
     * @return  bool    Success status
     */
    public function updateOrderStatus($orderId, $newStatus, $statusText, $amount)
    {
        try {
            $db = Factory::getContainer()->get('DatabaseDriver');
            
            // Update order status
            $query = $db->getQuery(true);
            $query->update('#__virtuemart_orders')
                  ->set($db->quoteName('order_status') . ' = ' . $db->quote($newStatus))
                  ->where($db->quoteName('virtuemart_order_id') . ' = ' . (int)$orderId);
            
            $db->setQuery($query);
            $db->execute();
            
            // Update order history
            $this->updateOrderStatusHistory($orderId, $newStatus, $statusText, $amount);
            
            
            return true;
            
        } catch (\Exception $e) {
            // Error updating order status - silent fail
            return false;
        }
    }

    /**
     * Update order status history
     */
    private function updateOrderStatusHistory($orderId, $newStatus, $statusText, $amount)
    {
        try {
            $db = Factory::getContainer()->get('DatabaseDriver');
            
            $comment = 'Refunded processed via iCard - Amount: ' . number_format($amount, 2);
            
            $query = $db->getQuery(true);
            $query->insert('#__virtuemart_order_histories')
                  ->set($db->quoteName('virtuemart_order_id') . ' = ' . (int)$orderId)
                  ->set($db->quoteName('order_status_code') . ' = ' . $db->quote($newStatus))
                  ->set($db->quoteName('comments') . ' = ' . $db->quote($comment))
                  ->set($db->quoteName('created_on') . ' = NOW()')
                  ->set($db->quoteName('created_by') . ' = 1');
            
            $db->setQuery($query);
            $db->execute();
            
            
        } catch (\Exception $e) {
            // Error updating order status history - silent fail
        }
    }



    /**
     * Get transaction history for an order
     *
     * @param   int     $virtuemart_order_id  Order ID
     * @param   object  $paymentTable         Payment table data
     *
     * @return  array   Array of transactions
     */
    public function getTransactionHistory($virtuemart_order_id, $paymentTable)
    {
        $transactions = [];
        
        // Add original payment transaction
        if (!empty($paymentTable->icard_transaction_id)) {
            $transactions[] = [
                'type' => 'Payment',
                'transaction_id' => $paymentTable->icard_transaction_id,
                'date' => $paymentTable->icard_payment_date ?? 'N/A',
                'amount' => '+' . number_format($paymentTable->payment_order_total, 2) . ' ' . $this->getCurrencyCode($paymentTable->payment_currency),
                'color' => '#28a745'
            ];
        }
        
        // Add individual refund transactions
        if (!empty($paymentTable->icard_refund_transaction_id)) {
            $refundIds = explode(',', $paymentTable->icard_refund_transaction_id);
            $refundAmounts = !empty($paymentTable->icard_refund_amounts) ? explode(',', $paymentTable->icard_refund_amounts) : [];
            $refundDates = !empty($paymentTable->icard_refund_dates) ? explode(',', $paymentTable->icard_refund_dates) : [];
            
            foreach ($refundIds as $index => $refundId) {
                $amount = isset($refundAmounts[$index]) ? $refundAmounts[$index] : 0;
                $date = isset($refundDates[$index]) ? $refundDates[$index] : ($paymentTable->icard_refund_date ?? 'N/A');
                
                $transactions[] = [
                    'type' => 'Refund ' . ($index + 1),
                    'transaction_id' => trim($refundId),
                    'date' => $date,
                    'amount' => '-' . number_format($amount, 2) . ' ' . $this->getCurrencyCode($paymentTable->payment_currency),
                    'color' => '#dc3545'
                ];
            }
        }
        
        return $transactions;
    }

    /**
     * Get currency code
     */
    private function getCurrencyCode($currencyId)
    {
        try {
            $db = Factory::getContainer()->get('DatabaseDriver');
            $query = $db->getQuery(true)
                ->select('currency_code_3')
                ->from('#__virtuemart_currencies')
                ->where($db->quoteName('virtuemart_currency_id') . ' = ' . (int)$currencyId);
            $db->setQuery($query);
            return $db->loadResult() ?: 'EUR';
        } catch (\Exception $e) {
            return 'EUR';
        }
    }
}

