<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  Vmpayment.Icard
 *
 * @copyright   Copyright (C) 2025 iCard AD. All rights reserved.
 * @license     GNU General Public License version 3 or later; see http://www.gnu.org/licenses/gpl-3.0.html
 */

namespace Icard\Plugin\Vmpayment\Icard\Extension;

defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Icard\Plugin\Vmpayment\Icard\Service\RefundService;
use Icard\Plugin\Vmpayment\Icard\Service\GatewayTestService;
use Icard\Plugin\Vmpayment\Icard\Service\SignatureService;
use Icard\Plugin\Vmpayment\Icard\Repository\RefundRepository;
use Icard\Plugin\Vmpayment\Icard\UI\RefundUIRenderer;

// Load VirtueMart defines and constants
if (!function_exists('loadVirtueMartDefines')) {
    function loadVirtueMartDefines() {
        if (!defined('JPATH_VM_PLUGINS')) {
            // Load Joomla legacy compatibility for VirtueMart 4.4.4
            if (!class_exists('JFactory')) {
                require_once JPATH_ROOT . '/libraries/joomla/legacy/factory.php';
            }
            
            $vmdefines_path = JPATH_ADMINISTRATOR . '/components/com_virtuemart/helpers/vmdefines.php';
            if (file_exists($vmdefines_path)) {
                require_once $vmdefines_path;
                \vmDefines::defines();
            }
        }
    }
}
loadVirtueMartDefines();



/**
 * iCard Payment Plugin for VirtueMart (Joomla 5.4)
 * Modern namespaced version
 * 
 * DEPENDENCIES:
 * - Joomla 5.4
 * - VirtueMart 4.4.4 (e-commerce component) - REQUIRED
 * - PHP 8.1+ with OpenSSL extension
  
 * The plugin extends vmPSPlugin and uses VirtueMart-specific:
 * - Database tables (#__virtuemart_orders, #__virtuemart_paymentmethods)
 * - Event handlers (plgVmConfirmedOrder, plgVmOnPaymentNotification)
 * - Order management system
 * - Payment processing workflow
 * 
 * @package     Icard.Plugin
 * @subpackage  Vmpayment.Icard
 * @since       1.0.0
 */
class Icard extends \vmPSPlugin
{
    /**
     * Service instances
     */
    private $refundRepository;
    private $refundService;
    private $refundUIRenderer;
    private $gatewayTestService;
    private $signatureService;

    /**
     * Constructor - Initialize plugin
     *
     * @param   mixed  $dispatcher  The dispatcher
     * @param   array  $config      Configuration array
     */
    public function __construct($dispatcher, array $config = [])
    {
        // Call parent constructor with legacy parameters for VirtueMart compatibility
        parent::__construct($dispatcher, $config);

        // Load language files
        $lang = Factory::getApplication()->getLanguage();
        $lang->load('plg_vmpayment_icard', JPATH_ADMINISTRATOR, null, true);
        
        // Configure plugin
        $this->_loggable = true;
        $this->_debug = Factory::getApplication()->get('debug', false);
        $this->tableFields = array_keys($this->getTableSQLFields());
        $this->_tablepkey = 'id';
        $this->_tableId = 'id';
        
        // Configure VirtueMart table properties for getVmPluginMethod
        $this->_configTable = '#__virtuemart_paymentmethods';
        $this->_configTableFileName = 'paymentmethods';
        $this->_configTableClassName = 'TablePaymentmethods';
        $this->_configTableFieldName = 'payment_params';
        $this->_xParams = 1;
                
        // Configuration parameters
        $varsToPush = [
            'test' => ['', 'int'],
            
            // Production credentials
            'production_merchant_id' => ['', 'char'],
            'production_client_id' => ['', 'char'],
            'production_currency' => ['', 'char'],
            'production_private_key' => ['', 'char'],
            'production_public_key' => ['', 'char'],
            'production_key_index' => ['', 'char'],
            'production_key_response_index' => ['', 'char'],
            'production_url' => ['', 'char'],

            // Development credentials
            'developer_merchant_id' => ['', 'char'],
            'developer_client_id' => ['', 'char'],
            'developer_currency' => ['', 'char'],
            'developer_private_key' => ['', 'char'],
            'developer_public_key' => ['', 'char'],
            'developer_key_index' => ['', 'char'],
            'developer_key_response_index' => ['', 'char'],
            'developer_url' => ['', 'char'],
            
            // Refund configuration
            'allow_refunds' => [1, 'int'],
        ];
        
        $this->addVarsToPushCore($varsToPush, 1);
        $this->setConfigParameterable($this->_configTableFieldName, $varsToPush);
        
        // Set the varsToPushParam for getVmPluginMethod
        $this->_varsToPushParam = $varsToPush;

        // Add CSS and JS resources
        $document = Factory::getApplication()->getDocument();
        $document->addStyleSheet(Uri::root() . '/plugins/vmpayment/icard/joomlaadmin.css');
        
        // Load test connection JS in admin only
        $app = Factory::getApplication();
        if ($app->isClient('administrator')) {
            // Load ToastService first
            $document->addScript(Uri::root() . '/plugins/vmpayment/icard/src/Service/ToastService.js');
            $document->addScript(Uri::root() . '/plugins/vmpayment/icard/admin.js');
        }

        // Initialize service classes
        $this->initializeServices();
    }


    /**
     * Initialize service classes for refund functionality
     *
     * @return  void
     */
    private function initializeServices()
    {
        // Initialize repository with table name
        $this->refundRepository = new RefundRepository(
            $this->_tablename
        );
        
        // Initialize signature service with logger callback
        $this->signatureService = new SignatureService([$this, 'log_to_file']);
        
        // Initialize refund service with repository, logger, and signature service
        $this->refundService = new RefundService(
            $this->refundRepository,
            [$this, 'log_to_file'],
            $this->signatureService
        );
        
        // Initialize UI renderer with repository
        $this->refundUIRenderer = new RefundUIRenderer(
            $this->refundRepository
        );
        
        // Initialize gateway test service with plugin instance for logging
        $this->gatewayTestService = new GatewayTestService($this);
    }

    /**
     * Log message using VirtueMart's standard logging
     * Forces logging to main com_virtuemart.log.php file
     *
     * @param   string  $message  Message to log
     * @param   string  $level    Log level
     *
     * @return  void
     */
    public function log_to_file($message, $level = 'INFO')
    {
        // Force logging to main VirtueMart log file instead of icard.0.log.php
        if (!class_exists('vmEcho')) {
            require_once(VMPATH_ADMIN . '/helpers/vmecho.php');
        }
        $oldLogFileName = \vmEcho::$logFileName;
        \vmEcho::$logFileName = 'com_virtuemart';
        \logInfo($message, $level);
        \vmEcho::$logFileName = $oldLogFileName;
    }

    /**
     * Define database table structure
     */
    function getTableSQLFields()
    {
        return [
            'id' => 'int(11) UNSIGNED NOT NULL AUTO_INCREMENT',
            'virtuemart_order_id' => 'int(1) UNSIGNED',
            'order_number' => 'char(64)',
            'virtuemart_paymentmethod_id' => 'mediumint(1) UNSIGNED',
            'payment_name' => 'varchar(5000)',
            'payment_order_total' => 'decimal(15,5) NOT NULL DEFAULT \'0.00000\'',
            'payment_currency' => 'char(3)',
            'cost_per_transaction' => 'decimal(10,2)',
            'cost_percent_total' => 'decimal(10,2)',
            'tax_id' => 'smallint(1)',
            'user_session' => 'varchar(255)',
            
            // iCard response data
            'icard_merchant_id' => 'varchar(50)',
            'icard_transaction_id' => 'varchar(50)',
            'icard_amount' => 'decimal(19,2)',
            'icard_currency' => 'char(3)',
            'icard_status' => 'varchar(20)',
            'icard_response_code' => 'varchar(10)',
            'icard_response_message' => 'varchar(255)',
            'icard_payment_date' => 'datetime',
            'icardresponse_raw' => 'varchar(512)',
            
            // Refund tracking fields
            'icard_refund_transaction_id' => 'text',
            'icard_refund_amount' => 'decimal(19,2)',
            'icard_refund_date' => 'datetime',
            'icard_refund_status' => 'varchar(20)',
            'icard_refund_reason' => 'varchar(255)',
            'icard_total_refunded' => 'decimal(19,2) DEFAULT 0.00',
            'icard_refund_count' => 'int DEFAULT 0',
            'icard_refund_amounts' => 'text',
            'icard_refund_dates' => 'text',
            
            // Additional fields
            'last_check' => 'int NULL',
            'expired_time' => 'int NOT NULL',
            'test_environment' => 'boolean NOT NULL'
        ];
    }

    /**
     * Create database table
     */
    public function getVmPluginCreateTableSQL()
    {
        return $this->createTableSQL('Payment icard Table');
    }

    /**
     * Joomla plugin uninstallation - Clean up orphaned configurations
     * 
     * Called by Joomla when the plugin is uninstalled from Extension Manager.
     * 
     * @param   object  $parent  The parent installer object
     * @return  bool    Success status
     */
    public function onUninstall($parent)
    {
        // Clean up orphaned payment method configurations
        $this->cleanupOrphanedPaymentMethods();
        
        return true;
    }

    /**
     * Clean up orphaned payment method configurations
     */
    private function cleanupOrphanedPaymentMethods()
    {
        try {
            $db = Factory::getContainer()->get('DatabaseDriver');
            
            // Find and remove orphaned iCard payment methods
            $query = $db->getQuery(true);
            $query->delete('#__virtuemart_paymentmethods')
                  ->where($db->quoteName('payment_element') . ' = ' . $db->quote('icard'));
            
            $db->setQuery($query);
            $db->execute();
            
        } catch (\Exception $e) {
            // Silent fail - don't break uninstallation
        }
    }

    /**
     * Handle confirmed order - redirect to iCard gateway
     */
    function plgVmConfirmedOrder($cart, $order)
    {
        if (!($method = $this->getVmPluginMethod($order['details']['BT']->virtuemart_paymentmethod_id))) {
            return null;
        }

        if (!$this->selectedThisElement($method->payment_element)) {
            return false;
        }

        $session = Factory::getApplication()->getSession();
        $return_context = $session->getId();

        if (!class_exists('VirtueMartModelOrders')) {
            require(VMPATH_ADMIN . DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'orders.php');
        }
        if (!class_exists('VirtueMartModelCurrency')) {
            require(VMPATH_ADMIN . DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'currency.php');
        }

        $address = isset($order['details']['ST']) ? $order['details']['ST'] : $order['details']['BT'];

        $totalInPaymentCurrency = \vmPSPlugin::getAmountInCurrency($order['details']['BT']->order_total, $method->payment_currency);

        if ($totalInPaymentCurrency['value'] <= 0) {
            \vmInfo(\vmText::_('VMPAYMENT_icard_PAYMENT_AMOUNT_INCORRECT'));
            return false;
        }

        $dbValues = [
            'user_session' => $return_context,
            'order_number' => $order['details']['BT']->order_number,
            'payment_name' => $this->renderPluginName($method),
            'virtuemart_paymentmethod_id' => $cart->virtuemart_paymentmethod_id,
            'cost_per_transaction' => $method->cost_per_transaction,
            'cost_percent_total' => $method->cost_percent_total,
            'payment_currency' => $method->payment_currency,
            'payment_order_total' => $totalInPaymentCurrency['value'],
            'tax_id' => $method->tax_id
        ];

        $this->storePSPluginInternalData($dbValues);

        // Save cart contents to session for potential restoration (avoid closure serialization)
        $session = Factory::getApplication()->getSession();
        $cart_data = [];
        foreach ($cart->products as $product) {
            $cart_data[] = [
                'virtuemart_product_id' => $product->virtuemart_product_id,
                'quantity' => $product->quantity,
                'product_name' => $product->product_name
            ];
        }
        $session->set('icard_backup_cart', json_encode($cart_data));
        $session->set('icard_backup_order_number', $order['details']['BT']->order_number);

        // Generate payment form
        $html = $this->getPostParameters($cart, $method, $address, $order, round($totalInPaymentCurrency['value'], 2));
        \vRequest::setVar('html', $html);
    }

    /**
     * Generate gateway payment form with all required parameters
     */
    private function getPostParameters($cart, $method, $address, $order, $order_total)
    {
        // NOTIFICATION URL (server-to-server callback from iCard gateway)
        $notify_url = Uri::root() .
            'index.php?option=com_virtuemart&view=pluginresponse&task=pluginnotification&tmpl=component&pm=' .
            $order['details']['BT']->virtuemart_paymentmethod_id;

        // SUCCESS URL (customer redirect) - Go directly to order details page
        $ok_url = Uri::root() .
            'index.php?option=com_virtuemart&view=orders&layout=details&order_number=' . 
            $order['details']['BT']->order_number . '&order_pass=' . $order['details']['BT']->order_pass;

        // CANCEL URL (customer cancels payment) - Must trigger plgVmOnUserPaymentCancel()
        $cancel_url = Uri::root() .
            'index.php?option=com_virtuemart&view=pluginresponse&task=pluginUserPaymentCancel&pm=' .
            $order['details']['BT']->virtuemart_paymentmethod_id . '&on=' . $order['details']['BT']->order_number;

        // Get credentials based on test mode
        $test = $method->test;
        if (!$test) {
            $merchant_id = $method->production_merchant_id;
            $client_id = $method->production_client_id;
            $currency = $method->production_currency;
            $private_key = $method->production_private_key;
            $public_key = $method->production_public_key;
            $key_index = $method->production_key_index;
            $key_response_index = $method->production_key_response_index;
            $url = $method->production_url;
        } else {
            $merchant_id = $method->developer_merchant_id;
            $client_id = $method->developer_client_id;
            $currency = $method->developer_currency;
            $private_key = $method->developer_private_key;
            $public_key = $method->developer_public_key;
            $key_index = $method->developer_key_index;
            $key_response_index = $method->developer_key_response_index;
            $url = $method->developer_url;
        }

        // Build iCard API parameters
        $post = [
            'IPGmethod' => 'IPGPurchase',
            'IPGVersion' => '4.5',
            'Language' => 'en',
            'KeyIndex' => $key_index,
            'KeyIndexResp' => $key_response_index,
            'MID' => $merchant_id,
            'MIDName' => 'iCard Checkout',
            'Originator' => $client_id,
            'BannerIndex' => 8,
            'Amount' => number_format($order_total, 2, '.', ''),
            'Currency' => $currency,
            'OrderID' => str_pad($order['details']['BT']->virtuemart_order_id, 6, "0", STR_PAD_LEFT) . '_' . $this->generateUuid(),
            'CustomerIP' => isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '',
            'Email' => (!isset($address->email) ? $order['details']['BT']->email : $address->email),
            'MobileNumber' => $address->phone_1,
            'BillAddrCountry' => \ShopFunctions::getCountryByID($address->virtuemart_country_id, 'country_2_code'),
            'BillAddrPostCode' => $address->zip,
            'BillAddrCity' => $address->city,
            'BillAddrLine1' => $address->address_1,
            'URL_OK' => $ok_url,
            'URL_Cancel' => $cancel_url,
            'URL_Notify' => $notify_url
        ];

        $this->log_to_file('Payment form data: ' . json_encode($post, JSON_PRETTY_PRINT), 'INFO');

        // Create signature using centralized service
        $post['Signature'] = $this->signatureService->createSignature($post, $private_key);

        // Generate HTML form
        $form = "<br><br><form id='Payment_form' method='post' action='" . htmlspecialchars($url, ENT_QUOTES, 'UTF-8') . "'>";
        foreach ($post as $key => $value) {
            $form .= "<input type='hidden' name='" . htmlspecialchars($key, ENT_QUOTES, 'UTF-8') . "' value='" . htmlspecialchars($value, ENT_QUOTES, 'UTF-8') . "'/>";
        }
        $form .= "<noscript><input type='submit' value='Pay with iCard' class='button'></noscript>";
        $form .= "<br><br></form><br><br>";
        $form .= "<script type='text/javascript'>document.getElementById('Payment_form').submit();</script>";

        return $form;
    }

    /**
     * Handle order status updates - detect refund requests
     */
    function plgVmOnUpdateOrderPayment(&$order, $old_order_status)
    {
        // Handle TableOrders object - convert to array format for consistency
        if (is_object($order)) {
            // Get order ID from TableOrders object
            $order_id = $order->virtuemart_order_id;
            $payment_method_id = $order->virtuemart_paymentmethod_id;
            $new_status = $order->order_status;
            
            // Only process if this is an iCard payment
            if (!$this->selectedThisByMethodId($payment_method_id)) {
                return null;
            }
        } else {
            // Handle array format (legacy compatibility)
            if (!isset($order['details']['BT'])) {
                return null;
            }
            
            $order_id = $order['details']['BT']->virtuemart_order_id;
            $payment_method_id = $order['details']['BT']->virtuemart_paymentmethod_id;
            $new_status = $order['details']['BT']->order_status;
            
            // Only process if this is an iCard payment
            if (!$this->selectedThisElement($payment_method_id)) {
                return null;
            }
        }
        
        // Check if order status changed to refunded (all refunds use R status)
        if ($new_status === 'R' && $old_order_status !== 'R') {
            $this->_processRefund($order, $old_order_status);
        }
        
        return null;
    }

    /**
     * Process refund request - Delegates to RefundService
     *
     * @param   mixed   $order           Order data (object or array)
     * @param   string  $old_order_status  Previous order status
     *
     * @return  bool    Success status
     */
    private function _processRefund($order, $old_order_status)
    {
        try {
            // Handle both TableOrders object and array formats
            if (is_object($order)) {
                $virtuemart_order_id = $order->virtuemart_order_id;
                $payment_method_id = $order->virtuemart_paymentmethod_id;
            } else {
                $virtuemart_order_id = $order['details']['BT']->virtuemart_order_id;
                $payment_method_id = $order['details']['BT']->virtuemart_paymentmethod_id;
            }
            
            // Get payment table data
            $paymentTable = $this->_getInternalData($virtuemart_order_id);
            if (!$paymentTable) {
                return false;
            }

            // Get payment method configuration
            $method = $this->getVmPluginMethod($payment_method_id);
            if (!$method) {
                return false;
            }
            
            // Get refund amount from session (if set by admin)
            $session = Factory::getApplication()->getSession();
            $refundAmount = $session->get('icard_refund_amount_' . $virtuemart_order_id);
            $session->clear('icard_refund_amount_' . $virtuemart_order_id);
            
            // Process refund using service
            $result = $this->refundService->processRefund(
                $order,
                $paymentTable,
                $method,
                $refundAmount,
                $old_order_status
            );
            
            if ($result['success']) {
                // Determine if this is a full or partial refund
                $isFullRefund = $result['data']['is_full_refund'] ?? false;
                $newStatus = $isFullRefund ? 'R' : 'C'; // Only set to R if fully refunded
                $statusText = $isFullRefund ? 'Fully Refunded' : 'Partial Refund Applied';
                
                // Update order status using repository
                $this->refundRepository->updateOrderStatus(
                    $virtuemart_order_id,
                    $newStatus,
                    $statusText,
                    $result['data']['refund_amount']
                );
                
                return true;
            }
            
            return false;
            
        } catch (\Exception $e) {
            return false;
        }
    }

    // Old refund methods removed - now handled by RefundService and RefundRepository

    /**
     * Handle payment notifications from gateway
     */
    function plgVmOnPaymentNotification()
    {
        header('X-Content-Type-Options: nosniff');
        header('X-Frame-Options: DENY');
        header('X-XSS-Protection: 1; mode=block');
        header('Content-Type: text/plain; charset=utf-8');
        header('Connection: close');
        
        // Clear any output buffers to ensure immediate response
        if (ob_get_level()) {
            ob_end_clean();
        }
        
        // Set short timeout for quick response to prevent gateway timeout
        set_time_limit(5);
        ignore_user_abort(true);
        
        $app = Factory::getApplication();
        
        $virtuemart_paymentmethod_id = $app->input->getCmd('pm', 0);
        
        $method = $this->getVmPluginMethod($virtuemart_paymentmethod_id);
        if (!$method) {
            http_response_code(404);
            echo 'METHOD_NOT_FOUND';
            exit;
        }
        
        if (!$this->selectedThisElement($method->payment_element)) {
            echo 'OK';
            exit;
        }
        
        // SECURITY: Read and validate JSON input
        $raw_body = file_get_contents('php://input');
        
        // SECURITY: Limit input size to prevent DoS
        if (strlen($raw_body) > 10000) {
            http_response_code(413);
            echo 'PAYLOAD_TOO_LARGE';
            exit;
        }
        
        $payload = json_decode($raw_body, true);
        
        // SECURITY: Validate JSON structure and content
        if (empty($payload) || !is_array($payload)) {
            echo 'OK';
            exit;
        }
        
        // SECURITY: Validate required fields exist
        if (!isset($payload['Payment']) || !isset($payload['Operation'])) {
            http_response_code(400);
            echo 'INVALID_PAYLOAD';
            exit;
        }
        
        // CRITICAL SECURITY: Verify signature before processing
        $receivedSignature = $payload['Signature'] ?? $payload['signature'] ?? null;
        $publicKey = $method->test ? $method->developer_public_key : $method->production_public_key;
        
        if (!$this->signatureService->verifySignature($payload, $publicKey, $receivedSignature)) {
            http_response_code(400);
            echo 'INVALID_SIGNATURE';
            exit;
        }

        // Send OK immediately to prevent iCard timeout (within 2 seconds)
        echo 'OK';
        if (ob_get_level()) {
            ob_end_flush();
        }
        flush();
        
        // Close connection to client immediately
        if (function_exists('fastcgi_finish_request')) {
            fastcgi_finish_request();
        }
        
        // Extract data from iCard's nested structure (no fallbacks needed)
        $gateway_order_id = $payload['Payment']['OrderId'] ?? '';
        $status = $payload['Payment']['Status'] ?? '';
        $transaction_id = $payload['Operation']['Provider']['Trn'] ?? '';
        $error_message = $payload['Operation']['Message'] ?? '';
        $error_code = $payload['Operation']['Code'] ?? '';
        
        $invoice = intval($gateway_order_id);
        
        if ($invoice <= 0) {
            echo 'OK';
            exit;
        }

        if (!class_exists('VirtueMartModelOrders')) {
            require(JPATH_VM_ADMINISTRATOR . DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'orders.php');
        }
        
        // Check if order exists
        $orderModel = \VmModel::getModel('orders');
        $order = $orderModel->getOrder($invoice);
        if (!$order || !isset($order['details']['BT'])) {
            echo 'OK';
            exit;
        }

        $status_lower = strtolower($status);

        try {
            $this->_storeIcardResponseData($invoice, $payload, $gateway_order_id, $status, $transaction_id, $error_message, $error_code, $method);
        } catch (\Exception $e) {
            // Continue processing even if data storage fails
        }

        if ($status_lower == 'success') {
            try {
                // Set timeout for status update to prevent hanging
                $original_timeout = ini_get('max_execution_time');
                set_time_limit(30); // 30 seconds max for status update
                
                $order['order_status'] = 'C';
                $order['customer_notified'] = 1;
                $comment = 'iCard payment successful';
                if (!empty($transaction_id)) {
                    $comment .= '. Transaction ID: ' . $transaction_id;
                }
                $order['comments'] = $comment;
                
                // Use direct database update for faster, more reliable status updates
                try {
                    $this->refundRepository->updateOrderStatus($invoice, 'C', 'iCard payment successful', 0);
                } catch (\Exception $e) {
                    // Continue processing even if status update fails
                }
                
                // Restore original timeout
                set_time_limit($original_timeout);
                
            } catch (\Exception $e) {
                // Critical error in payment processing
            }
            
            exit;
        } else {
            try {
                $order['order_status'] = 'X';
                $order['customer_notified'] = 1;
                $comment = 'iCard payment failed';
                if (!empty($error_message)) {
                    $comment .= '. Error: ' . $error_message;
                }
                if (!empty($error_code)) {
                    $comment .= '. Code: ' . $error_code;
                }
                $order['comments'] = $comment;
                $orderModel->updateStatusForOneOrder($invoice, $order, true);
            } catch (\Exception $e) {
                // Error updating order status
            }
            
            exit;
        }
    }

    /**
     * Handle payment cancellation or declined payment
     */
    function plgVmOnUserPaymentCancel()
    {
        if (!class_exists('VirtueMartModelOrders')) {
            require(VMPATH_ADMIN . DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'orders.php');
        }

        $order_number = \vRequest::getString('on', '');
        $virtuemart_paymentmethod_id = \vRequest::getInt('pm', '');

        if (empty($order_number) or
            empty($virtuemart_paymentmethod_id) or
            !$this->selectedThisByMethodId($virtuemart_paymentmethod_id)
        ) {
            return null;
        }

        if (!($virtuemart_order_id = \VirtueMartModelOrders::getOrderIdByOrderNumber($order_number))) {
            return null;
        }

        if (!($paymentTable = $this->getDataByOrderId($virtuemart_order_id))) {
            return null;
        }

        // Get the specific error message from iCard
        $error_message = $this->_checkPaymentStatus($paymentTable, []);
        
        // Update order status to Cancelled
        try {
            $this->refundRepository->updateOrderStatus(
                $virtuemart_order_id,
                'X', // Cancelled status
                'Payment cancelled or declined by customer',
                0
            );
        } catch (\Exception $e) {
            // Error updating order status
        }
        
        // Restore cart and show specific error message
        $this->_restoreCartFromSession(false);
        
        $app = Factory::getApplication();
        $app->enqueueMessage($error_message, 'error');
        
        // Redirect to checkout
        $checkout_url = \Joomla\CMS\Router\Route::_("index.php?option=com_virtuemart&view=cart&checkout=1&icard_error=1&order_id=" . $virtuemart_order_id, false);
        
        $app->redirect($checkout_url);
    }

    /**
     * Check payment status and return error message if declined
     */
    private function _checkPaymentStatus($paymentTable, $order)
    {
        $error_message = '';
        
        if (!empty($paymentTable->icard_status) && $paymentTable->icard_status !== 'success') {
            if (!empty($paymentTable->icard_response_message)) {
                $error_message = 'Payment declined: ' . $paymentTable->icard_response_message;
            } else {
                $error_message = 'Payment was declined. Please try again or use a different payment method.';
            }
            
            if (!empty($paymentTable->icard_response_code)) {
                $error_message .= ' (Code: ' . $paymentTable->icard_response_code . ')';
            }
        }
        
        if (isset($order['details']['BT']->order_status)) {
            $order_status = $order['details']['BT']->order_status;
            
            if ($order_status === 'X' && empty($error_message)) {
                $error_message = 'Payment was cancelled or declined. Please try again.';
            }
        }
        
        return $error_message;
    }

    /**
     * Display stored payment error messages from database
     */
    private function _displayStoredPaymentMessages()
    {
        $app = Factory::getApplication();
        $input = $app->input;
        
        // Only run when there's an error flag to avoid unnecessary database queries
        if ($input->getCmd('icard_error') !== '1') {
            return;
        }
        
        // Check if this is an error return from gateway
        if ($input->getCmd('icard_error') === '1') {
            $order_id = $input->getInt('order_id', 0);
            
            $this->_restoreCartFromSession(false);
            
            if ($order_id > 0) {
                try {
                    $db = Factory::getContainer()->get('DatabaseDriver');
                    $query = $db->getQuery(true);
                    $query->select('order_status')
                          ->from('#__virtuemart_orders')
                          ->where($db->quoteName('virtuemart_order_id') . ' = ' . (int)$order_id)
                          ->where($db->quoteName('order_status') . ' = ' . $db->quote('X'));
                    
                    $db->setQuery($query);
                    $result = $db->loadResult();
                    
                    if (!empty($result)) {
                        $payment_query = $db->getQuery(true);
                        $payment_query->select('icard_response_message, icard_response_code')
                                      ->from($this->_tablename)
                                      ->where($db->quoteName('virtuemart_order_id') . ' = ' . (int)$order_id);
                        
                        $db->setQuery($payment_query);
                        $payment_data = $db->loadObject();
                        
                        if ($payment_data && !empty($payment_data->icard_response_message)) {
                            $display_message = 'Payment declined: ' . $payment_data->icard_response_message;
                            if (!empty($payment_data->icard_response_code)) {
                                $display_message .= ' (Code: ' . $payment_data->icard_response_code . ')';
                            }
                        } else {
                            $display_message = 'Payment was declined. Please try again or use a different payment method.';
                        }
                        
                        $app->enqueueMessage($display_message, 'error');
                    }
                    
                } catch (\Exception $e) {
                    $this->log_to_file('Error retrieving payment error from database: ' . $e->getMessage(), 'ERROR');
                }
            }
        }
    }

    /**
     * Restore cart from session backup
     */
    private function _restoreCartFromSession($clear_backup = true)
    {
        try {
            $session = Factory::getApplication()->getSession();
            $backup_cart = $session->get('icard_backup_cart');

            if (!empty($backup_cart)) {
                $cart_products = json_decode($backup_cart, true);
                
                if (is_array($cart_products) && !empty($cart_products)) {
                    $cart = \VirtueMartCart::getCart();
                    
                    // Clear current cart
                    $cart->emptyCart();
                    
                    // Restore products
                    foreach ($cart_products as $product) {
                        $cart->add($product['virtuemart_product_id'], $product['quantity']);
                    }
                }
            }
            
            // Only clear backup if requested (don't clear before redirect)
            if ($clear_backup) {
                $session->clear('icard_backup_cart');
                $session->clear('icard_backup_order_number');
            }
            
        } catch (\Exception $e) {
            // Error restoring cart from session
        }
    }

    /**
     * Store iCard response data in database
     */
    private function _storeIcardResponseData($virtuemart_order_id, $payload, $gateway_order_id, $status, $transaction_id, $error_message, $error_code, $method)
    {
        try {
            $db = Factory::getContainer()->get('DatabaseDriver');
            
            $merchant_id = '';
            $amount = '';
            $currency = '';
            
            if (isset($payload['Payment']['MID'])) {
                $merchant_id = $payload['Payment']['MID'];
            }
            if (isset($payload['Payment']['Sum']['Amount'])) {
                $amount = $payload['Payment']['Sum']['Amount'];
            }
            if (isset($payload['Payment']['Sum']['Currency'])) {
                $currency = $payload['Payment']['Sum']['Currency'];
            }
            
            $query = $db->getQuery(true);
            $query->update($this->_tablename)
                  ->set($db->quoteName('icard_merchant_id') . ' = ' . $db->quote($merchant_id))
                  ->set($db->quoteName('icard_transaction_id') . ' = ' . $db->quote($transaction_id))
                  ->set($db->quoteName('icard_amount') . ' = ' . $db->quote($amount))
                  ->set($db->quoteName('icard_currency') . ' = ' . $db->quote($currency))
                  ->set($db->quoteName('icard_status') . ' = ' . $db->quote($status))
                  ->set($db->quoteName('icard_response_code') . ' = ' . $db->quote($error_code))
                  ->set($db->quoteName('icard_response_message') . ' = ' . $db->quote($error_message))
                  ->set($db->quoteName('icard_payment_date') . ' = NOW()')
                  ->set($db->quoteName('icardresponse_raw') . ' = ' . $db->quote($status . '|' . $error_code . '|' . $error_message))
                  ->where($db->quoteName('virtuemart_order_id') . ' = ' . (int)$virtuemart_order_id);
            
            $db->setQuery($query);
            $db->execute();
        } catch (\Exception $e) {
            // Error storing iCard response data
        }
    }


    // ============================================================================
    // HELPER FUNCTIONS (RSA SIGNATURE) OPERATIONS
    // ============================================================================

    /**
     * Generate a UUID v4
     *
     * @return  string  UUID string
     */
    private function generateUuid()
    {
        $data = random_bytes(16);
        
        // Set version to 0100
        $data[6] = chr(ord($data[6]) & 0x0f | 0x40);
        
        // Set bits 6-7 to 10
        $data[8] = chr(ord($data[8]) & 0x3f | 0x80);
        
        return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
    }

    // ============================================================================
    // GATEWAY CONNECTION TESTING
    // ============================================================================

    /**
     * AJAX handler for testing iCard connection and refund operations
     * Called via: ?option=com_ajax&plugin=icard&group=vmpayment&format=json
     */
    public function onAjaxIcard()
    {
        $app = Factory::getApplication();

        // SECURITY: Check if in administrator client
        if (!$app->isClient('administrator')) {
            $this->log_to_file('AJAX: Not in administrator client', 'ERROR');
            return ['success' => false, 'message' => 'Access denied'];
        }   
        
        // SECURITY: Check user permissions
        $user = Factory::getApplication()->getIdentity();
        
        if (!$user->authorise('core.admin')) {
            $this->log_to_file('AJAX: User not authorized', 'ERROR');
            return ['success' => false, 'message' => 'Access denied'];
        }
        
        $task = $app->input->getCmd('task', 'test_connection');
        
        // SECURITY: CSRF token validation
        if (!\Joomla\CMS\Session\Session::checkToken()) {
            $this->log_to_file('AJAX: Invalid security token for task: ' . $task, 'ERROR');
            return ['success' => false, 'message' => 'Invalid security token'];
        }

        switch ($task) {
            case 'test_connection':
                return $this->gatewayTestService->handleTestConnection();
            case 'process_refund_direct':
                return $this->_handleProcessRefund();
            default:
                return ['success' => false, 'message' => 'Unknown task'];
        }
    }
    
    /**
     * Handle refund processing AJAX request - Delegates to RefundService
     *
     * @return  array  Result array
     */
    private function _handleProcessRefund()
    {
        $app = Factory::getApplication();
        
        // SECURITY: Sanitize and validate input
        $order_id = (int) $app->input->getInt('order_id', 0);
        $amount = (float) $app->input->getFloat('amount', 0);
        
        // SECURITY: Validate input ranges
        if ($order_id <= 0 || $order_id > 999999) {
            $this->log_to_file('AJAX refund - Invalid order ID: ' . $order_id, 'ERROR');
            return ['success' => false, 'message' => 'Invalid order ID'];
        }
        
        if ($amount <= 0 || $amount > 999999.99) {
            $this->log_to_file('AJAX refund - Invalid amount: ' . $amount, 'ERROR');
            return ['success' => false, 'message' => 'Invalid refund amount'];
        }
        
        try {
            // Get order data
            if (!class_exists('VirtueMartModelOrders')) {
                require(VMPATH_ADMIN . DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'orders.php');
            }
            
            $orderModel = \VmModel::getModel('orders');
            $order = $orderModel->getOrder($order_id);
            
            if (!$order || !isset($order['details']['BT'])) {
                $this->log_to_file('AJAX refund - Order not found: ' . $order_id, 'ERROR');
                return ['success' => false, 'message' => 'Order not found'];
            }
            
            // Get payment data
            $paymentTable = $this->_getInternalData($order_id);
            if (!$paymentTable) {
                $this->log_to_file('AJAX refund - Payment data not found for order: ' . $order_id, 'ERROR');
                return ['success' => false, 'message' => 'Payment data not found'];
            }
            
            // Get payment method
            $method = $this->getVmPluginMethod($order['details']['BT']->virtuemart_paymentmethod_id);
            if (!$method) {
                $this->log_to_file('AJAX refund - Payment method not found', 'ERROR');
                return ['success' => false, 'message' => 'Payment method not found'];
            }
            
            // Check if refunds are enabled
            if ($method->allow_refunds != 1) {
                $this->log_to_file('AJAX refund - Refunds disabled for payment method', 'ERROR');
                return ['success' => false, 'message' => 'Refunds are disabled for this payment method'];
            }
            
            // Process refund using service
            $result = $this->refundService->processRefund(
                $order,
                $paymentTable,
                $method,
                $amount
            );

            if ($result['success']) {
                // Determine if this is a full or partial refund
                $isFullRefund = $result['data']['is_full_refund'] ?? false;
                $newStatus = $isFullRefund ? 'R' : 'C'; // Only set to R if fully refunded
                $statusText = $isFullRefund ? 'Fully Refunded' : 'Partial Refund Applied';
                
                // Update order status using repository
                $this->refundRepository->updateOrderStatus(
                    $order_id,
                    $newStatus,
                    $statusText,
                    $amount
                );
                
                return [
                    'success' => true,
                    'message' => 'Refund processed successfully',
                    'transaction_id' => $result['data']['transaction_id'],
                    'new_status' => $result['data']['new_status'],
                    'amount' => $amount
                ];
            }

            $this->log_to_file('AJAX refund - Result: ' . json_encode($result), 'INFO');
            
            return ['success' => false, 'message' => $result['message']];
            
        } catch (\Exception $e) {
            $this->log_to_file('AJAX refund - Exception: ' . $e->getMessage(), 'ERROR');
            $this->log_to_file('AJAX refund - Stack trace: ' . $e->getTraceAsString(), 'ERROR');
            return ['success' => false, 'message' => 'Error processing refund: ' . $e->getMessage()];
        }
    }

    // ============================================================================
    // VIRTUEMART EVENT HANDLERS (VM3/VM4 Compatible)
    // ============================================================================
    // These functions are automatically called by VirtueMart at specific points
    // in the shopping/payment lifecycle. The naming convention (plgVm*) is
    // MANDATORY - VirtueMart uses it to discover and trigger these event handlers.

    /**
     * Get payment currency
     */
    static function getPaymentCurrency(&$method, $selectedUserCurrency = false)
    {
        if (empty($method->payment_currency)) {
            $vendor_model = \VmModel::getModel('vendor');
            $vendor = $vendor_model->getVendor($method->virtuemart_vendor_id);
            $method->payment_currency = $vendor->vendor_currency;
            return $method->payment_currency;
        } else {
            $vendor_model = \VmModel::getModel('vendor');
            $vendor_currencies = $vendor_model->getVendorAndAcceptedCurrencies($method->virtuemart_vendor_id);

            if (!$selectedUserCurrency) {
                if ($method->payment_currency == -1) {
                    $mainframe = Factory::getApplication();
                    $selectedUserCurrency = $mainframe->getUserStateFromRequest("virtuemart_currency_id", 'virtuemart_currency_id', \vRequest::getInt('virtuemart_currency_id', $vendor_currencies['vendor_currency']));
                } else {
                    $selectedUserCurrency = $method->payment_currency;
                }
            }

            $vendor_currencies['all_currencies'] = explode(',', $vendor_currencies['all_currencies']);
            if (in_array($selectedUserCurrency, $vendor_currencies['all_currencies'])) {
                $method->payment_currency = $selectedUserCurrency;
            } else {
                $method->payment_currency = $vendor_currencies['vendor_currency'];
            }

            return $method->payment_currency;
        }
    }

    /**
     * Get internal data by order ID
     */
    function _getInternalData($virtuemart_order_id, $order_number = '')
    {
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);
        $query->select('*')
              ->from($db->quoteName($this->_tablename));
        
        if ($order_number) {
            $query->where($db->quoteName('order_number') . ' = ' . $db->quote($order_number));
        } else {
            $query->where($db->quoteName('virtuemart_order_id') . ' = ' . (int)$virtuemart_order_id);
        }

        $db->setQuery($query);
        if (!($paymentTable = $db->loadObject())) {
            return '';
        }
        return $paymentTable;
    }

    /**
     * Plugin installation
     * 
     * Called by VirtueMart when the plugin is installed.
     * Creates payment table.
     * 
     * @param   int  $jplugin_id  Joomla plugin ID
     * @return  bool  Success status
     */
    function plgVmOnStoreInstallPaymentPluginTable($jplugin_id)
    {
        // Call parent method to create payment table
        $result = $this->onStoreInstallPluginTable();
        
        return $result;
    }

    /**
     * Check payment method selection
     */
    public function plgVmOnSelectCheckPayment($cart, &$msg)
    {
        return $this->OnSelectCheck($cart);
    }

    /**
     * Display payment methods in cart
     */
    public function plgVmDisplayListFEPayment(\VirtueMartCart $cart, int $selected = 0, &$htmlIn = [])
    {
        // Check for stored payment error messages and display them
        $this->_displayStoredPaymentMessages();
        
        // Clear cart backup after successful display (if error was shown)
        $app = Factory::getApplication();
        $input = $app->input;
        if ($input->getCmd('icard_error') === '1') {
            $session = Factory::getApplication()->getSession();
            $session->clear('icard_backup_cart');
            $session->clear('icard_backup_order_number');
        }
        
        return $this->displayListFE($cart, $selected, $htmlIn);
    }

    /**
     * Calculate payment price
     */
    public function plgVmonSelectedCalculatePricePayment($cart, array &$cart_prices, &$cart_prices_name)
    {
        return $this->onSelectedCalculatePrice($cart, $cart_prices, $cart_prices_name);
    }

    /**
     * Check automatic payment selection
     */
    function plgVmOnCheckAutomaticSelectedPayment($cart, &$paymentCounter, array $cart_prices = [])
    {
        return $this->onCheckAutomaticSelected($cart, $cart_prices, $paymentCounter);
    }

    /**
     * Show order details in frontend
     */
    public function plgVmOnShowOrderFEPayment($virtuemart_order_id, $virtuemart_paymentmethod_id, &$payment_name)
    {
        $this->onShowOrderFE($virtuemart_order_id, $virtuemart_paymentmethod_id, $payment_name);
    }

    /**
     * Show order details in print
     */
    function plgVmonShowOrderPrintPayment($order_number, $method_id)
    {
        return $this->onShowOrderPrint($order_number, $method_id);
    }

    /**
     * Show order details in backend - Delegates to RefundUIRenderer
     *
     * @param   int  $virtuemart_order_id   Order ID
     * @param   int  $payment_method_id     Payment method ID
     *
     * @return  string|null  HTML output
     */
    function plgVmOnShowOrderBEPayment($virtuemart_order_id, $payment_method_id)
    {
        if (!$this->selectedThisByMethodId($payment_method_id)) {
            return null;
        }

        if (!($paymentTable = $this->_getInternalData($virtuemart_order_id))) {
            return '';
        }

        $this->getPaymentCurrency($paymentTable);
        
        // Get order data
        if (!class_exists('VirtueMartModelOrders')) {
            require(VMPATH_ADMIN . DIRECTORY_SEPARATOR . 'models' . DIRECTORY_SEPARATOR . 'orders.php');
        }

        $orderModel = \VmModel::getModel('orders');
        $order = $orderModel->getOrder($virtuemart_order_id);
        
        if (!$order || !isset($order['details']['BT'])) {
            return '';
        }
        
        $method = $this->getVmPluginMethod($payment_method_id);
        if (!$method) {
            return '';
        }
        
        // Render refund panel using UI renderer
        $html = $this->refundUIRenderer->renderRefundPanel(
            $virtuemart_order_id,
            $paymentTable,
            $payment_method_id,
            $order,
            $method
        );
        
        // Render transaction table using UI renderer
        $html .= $this->refundUIRenderer->renderTransactionTable(
            $virtuemart_order_id,
            $paymentTable
        );
        
        return $html;
    }


    /**
     * Declare plugin parameters for VM3
     */
    function plgVmDeclarePluginParamsPaymentVM3(&$data)
    {
        return $this->declarePluginParams('payment', $data);
    }

    /**
     * Set plugin parameters on table
     */
    function plgVmSetOnTablePluginParamsPayment($name, $id, &$table)
    {
        return $this->setOnTablePluginParams($name, $id, $table);
    }

    /**
     * Check payment conditions for iCard automatically
     */
    protected function checkConditions($cart, $method, $cart_prices)
    {
        $this->convert_condition_amount($method);

        $address = (($cart->ST == 0) ? $cart->BT : $cart->ST);

        $amount = $this->getCartAmount($cart_prices);
        $amount_cond = ($amount >= $method->min_amount and $amount <= $method->max_amount
            or
            ($method->min_amount <= $amount and ($method->max_amount == 0)));

        $countries = [];
        if (!empty($method->countries)) {
            if (!is_array($method->countries)) {
                $countries[0] = $method->countries;
            } else {
                $countries = $method->countries;
            }
        }

        // probably did not gave his BT:ST address
        if (!is_array($address)) {
            $address = [];
            $address['virtuemart_country_id'] = 0;
        }

        if (!isset($address['virtuemart_country_id'])) {
            $address['virtuemart_country_id'] = 0;
        }
        if (in_array($address['virtuemart_country_id'], $countries) || count($countries) == 0) {
            if ($amount_cond) {
                return true;
            }
        }

        return false;
    }
}
    