/**
 * iCard Admin Panel - Combined functionality
 * Handles test connection, mode validation, and admin UI interactions
 * Uses centralized ToastService for notifications
 */
(function() {
    'use strict';
    
    // Prevent multiple executions
    if (window.icardAdminScriptLoaded) {
        console.log('iCard admin script already loaded, preventing duplicate execution');
        return;
    }
    window.icardAdminScriptLoaded = true;
    
    // Wait for ToastService to be available
    function waitForToastService(callback) {
        if (window.IcardToastService) {
            callback();
        } else {
            setTimeout(() => waitForToastService(callback), 100);
        }
    }
    
    // Wrapper functions for backward compatibility
    function showToast(message, type = 'info', duration = 5000) {
        // Sanitize message to prevent null/undefined values
        if (!message || message === 'null' || message === 'undefined') {
            message = 'Unknown error occurred';
        }
        
        // If ToastService is already available, return the toast immediately
        if (window.IcardToastService) {
            return window.IcardToastService.show(message, type, { duration: duration });
        }
        
        // If ToastService is not ready, wait for it and return the toast
        var toastElement = null;
        waitForToastService(() => {
            toastElement = window.IcardToastService.show(message, type, { duration: duration });
        });
        return toastElement;
    }
    
    // ============================================================================
    // TEST CONNECTION FUNCTIONALITY
    // ============================================================================
    
    function testIcardConnection(event) {
        event.preventDefault();
        
        var button = event.target;
        
        // Check if test mode is disabled
        const testModeDisabled = document.querySelector('input[name="params[test]"][value="0"]:checked');
        if (testModeDisabled) {
            showToast('Test connection is disabled in production mode', 'warning');
            return;
        }
        
        // Get test mode from radio buttons
        var testMode = getTestMode();
        
        // Show loading state
        var loadingToast = showToast('Testing connection to iCard gateway...', 'loading', 0);
        resetButton(button);
        button.disabled = true;
        button.innerHTML = 'Testing...';
        
        // Make AJAX request
        var xhr = new XMLHttpRequest();
        xhr.open('POST', 'index.php?option=com_ajax&plugin=icard&group=vmpayment&format=json', true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
        xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');
        
        xhr.onreadystatechange = function() {
            if (xhr.readyState === 4) {
                button.disabled = false;
                button.innerHTML = 'Test Connection';
                
                if (xhr.status === 200) {
                    try {
                        var responseText = xhr.responseText;
                        if (!responseText || responseText.trim() === '') {
                            // Remove loading toast
                            if (loadingToast && loadingToast.parentElement) {
                                loadingToast.remove();
                            }
                            showToast('Empty response from server', 'error');
                            return;
                        }
                        
                        var response = JSON.parse(responseText);
                        handleTestResponse(response, testMode, loadingToast);
                    } catch (e) {
                        console.error('AJAX Error:', e);
                        console.error('Response:', xhr.responseText);
                        // Remove loading toast
                        if (loadingToast && loadingToast.parentElement) {
                            loadingToast.remove();
                        }
                        showToast('Invalid response from server: ' + e.message, 'error');
                    }
                } else {
                    // Remove loading toast
                    if (loadingToast && loadingToast.parentElement) {
                        loadingToast.remove();
                    }
                    showToast('Connection test failed: HTTP ' + xhr.status, 'error');
                }
            }
        };
        
        // Send request with test mode and CSRF token
        var csrfToken = getCSRFToken();
        var params = 'task=test_connection&test_mode=' + testMode + '&' + csrfToken + '=1';
        xhr.send(params);
        
        // Fallback timeout to remove loading toast after 30 seconds
        setTimeout(function() {
            if (loadingToast && loadingToast.parentElement) {
                loadingToast.remove();
                showToast('Connection test timed out', 'error');
            }
        }, 30000);
    }
    
    function handleTestResponse(response, testMode, loadingToast) {
        var modeText = testMode == 1 ? 'Development' : 'Production';
        
        // Remove loading toast first
        if (loadingToast && loadingToast.parentElement) {
            loadingToast.remove();
        }
        
        // Handle Joomla AJAX response format
        var actualResponse = response;
        if (response.data && response.data.length > 0) {
            actualResponse = response.data[0];
        }
        
        var message = actualResponse.message || 'Connection test completed';
        var success = actualResponse.success || false;
        
        if (success) {
            showToast(modeText + ' mode: ' + message, 'success');
        } else {
            showToast(modeText + ' mode: ' + message, 'error');
        }
    }
    
    function resetButton(button) {
        button.style.opacity = '1';
        button.style.transform = 'none';
    }
    
    function getTestMode() {
        var testModeRadios = document.querySelectorAll('input[name="params[test]"]');
        for (var i = 0; i < testModeRadios.length; i++) {
            if (testModeRadios[i].checked) {
                return testModeRadios[i].value;
            }
        }
        return 1; // Default to test mode
    }
    
    // ============================================================================
    // TEST MODE VALIDATION FUNCTIONALITY
    // ============================================================================
    
    function initializeTestModeValidation() {
        // Check test mode radio button
        const testModeRadios = document.querySelectorAll('input[name="params[test]"]');
        if (testModeRadios.length === 0) return;

        // Add change listeners to test mode radio buttons
        testModeRadios.forEach(radio => {
            radio.addEventListener('change', handleTestModeChange);
        });

        // Initial check
        handleTestModeChange();
    }

    function handleTestModeChange() {
        const testModeEnabled = document.querySelector('input[name="params[test]"][value="1"]:checked');
        const testModeDisabled = document.querySelector('input[name="params[test]"][value="0"]:checked');
        
        if (testModeDisabled) {
            // Production mode - check credentials and show warnings
            checkProductionCredentials();
            showProductionWarning();
            disableTestFields();
        } else if (testModeEnabled) {
            // Test mode - enable test fields
            hideProductionWarning();
            enableTestFields();
        }
    }

    function checkProductionCredentials() {
        const requiredFields = [
            'production_merchant_id',
            'production_client_id', 
            'production_currency',
            'production_private_key',
            'production_public_key',
            'production_key_index',
            'production_key_response_index',
            'production_url'
        ];
        
        const missingFields = [];
        
        requiredFields.forEach(fieldName => {
            const field = document.querySelector(`input[name="${fieldName}"], textarea[name="${fieldName}"]`);
            if (field && !field.value.trim()) {
                missingFields.push(getFieldLabel(fieldName));
            }
        });
        
        if (missingFields.length > 0) {
            showMissingCredentialsWarning(missingFields);
        } else {
            hideMissingCredentialsWarning();
        }
    }

    function getFieldLabel(fieldName) {
        const labelMap = {
            'production_merchant_id': 'Merchant ID',
            'production_client_id': 'Client ID',
            'production_currency': 'Currency',
            'production_private_key': 'Private Key',
            'production_public_key': 'Public Key',
            'production_key_index': 'Key Index',
            'production_key_response_index': 'Key Response Index',
            'production_url': 'Gateway URL'
        };
        
        return labelMap[fieldName] || fieldName;
    }

    function showMissingCredentialsWarning(missingFields) {
        // Remove existing warning
        hideMissingCredentialsWarning();
        
        const warningDiv = document.createElement('div');
        warningDiv.id = 'icard-production-warning';
        warningDiv.className = 'icard-production-warning show';
        warningDiv.innerHTML = `
            <strong>⚠️ Production Mode Active - Missing Credentials</strong><br>
            The following production credentials are required but missing:<br>
            <strong>${missingFields.join(', ')}</strong><br>
            <small>Please fill in all production credentials to use the payment gateway in production mode.</small>
        `;
        
        // Insert after the test mode section
        const testModeSection = document.querySelector('#params_test').closest('.control-group');
        if (testModeSection) {
            testModeSection.insertAdjacentElement('afterend', warningDiv);
        }
    }

    function hideMissingCredentialsWarning() {
        const existingWarning = document.getElementById('icard-production-warning');
        if (existingWarning) {
            existingWarning.remove();
        }
    }

    function showProductionWarning() {
        const warningDiv = document.createElement('div');
        warningDiv.id = 'icard-production-mode-warning';
        warningDiv.className = 'icard-production-warning show';
        warningDiv.innerHTML = `
            <strong>🚨 PRODUCTION MODE ACTIVE</strong><br>
            <strong>Warning:</strong> You are now in production mode. All transactions will be processed with real money.<br>
            <small>Make sure all production credentials are correctly configured before processing live payments.</small>
        `;
        
        // Insert at the top of the form
        const form = document.querySelector('form[name="adminForm"]');
        if (form) {
            form.insertBefore(warningDiv, form.firstChild);
        }
    }

    function hideProductionWarning() {
        const existingWarning = document.getElementById('icard-production-mode-warning');
        if (existingWarning) {
            existingWarning.remove();
        }
    }

    function disableTestFields() {
        const testFields = document.querySelectorAll('input[name*="developer_"], textarea[name*="developer_"]');
        testFields.forEach(field => {
            field.closest('.control-group').classList.add('icard-test-disabled');
        });
        
        const testButton = document.getElementById('icard-test-btn-dev');
        const testSection = document.querySelector('.icard-test-section');
        
        if (testButton) {
            testButton.disabled = true;
            testButton.style.opacity = '0.5';
            testButton.style.cursor = 'not-allowed';
        }
        
        if (testSection) {
            testSection.classList.add('icard-test-disabled');
        }
    }

    function enableTestFields() {
        const testFields = document.querySelectorAll('input[name*="developer_"], textarea[name*="developer_"]');
        testFields.forEach(field => {
            field.closest('.control-group').classList.remove('icard-test-disabled');
        });
        
        const testButton = document.getElementById('icard-test-btn-dev');
        const testSection = document.querySelector('.icard-test-section');
        
        if (testButton) {
            testButton.disabled = false;
            testButton.style.opacity = '1';
            testButton.style.cursor = 'pointer';
        }
        
        if (testSection) {
            testSection.classList.remove('icard-test-disabled');
        }
    }

    function addProductionFieldValidation() {
        const productionFields = document.querySelectorAll('input[name*="production_"], textarea[name*="production_"]');
        productionFields.forEach(field => {
            field.addEventListener('blur', function() {
                // Re-check production credentials when fields change
                const testModeDisabled = document.querySelector('input[name="params[test]"][value="0"]:checked');
                if (testModeDisabled) {
                    checkProductionCredentials();
                }
            });
        });
    }
    
    // ============================================================================
    // CSRF TOKEN HANDLING
    // ============================================================================
    
    function getCSRFToken() {
        // Method 1: Get CSRF token from meta tag
        var token = document.querySelector('meta[name="csrf-token"]');
        if (token) {
            return token.getAttribute('content');
        }
        
        // Method 2: Try to get from form
        var form = document.querySelector('form[name="adminForm"]');
        if (form) {
            var tokenInput = form.querySelector('input[name*="token"]');
            if (tokenInput) {
                return tokenInput.name;
            }
        }
        
        // Method 3: Try to get from any form
        var allForms = document.querySelectorAll('form');
        for (var i = 0; i < allForms.length; i++) {
            var tokenInput = allForms[i].querySelector('input[name*="token"]');
            if (tokenInput) {
                return tokenInput.name;
            }
        }
        
        // Method 4: Try to get from Joomla's global token
        if (window.Joomla && window.Joomla.getOptions && window.Joomla.getOptions('csrf.token')) {
            return window.Joomla.getOptions('csrf.token');
        }
        
        console.warn('CSRF token not found, using fallback');
        return 'token';
    }
    
    function addCSRFToken() {
        // Get CSRF token from Joomla
        var token = document.querySelector('meta[name="csrf-token"]');
        if (token) {
            // Add token to any forms that might need it
            var forms = document.querySelectorAll('form');
            forms.forEach(function(form) {
                if (!form.querySelector('input[name="' + token.getAttribute('content') + '"]')) {
                    var input = document.createElement('input');
                    input.type = 'hidden';
                    input.name = token.getAttribute('content');
                    input.value = token.getAttribute('content');
                    form.appendChild(input);
                }
            });
        }
    }
    
    // ============================================================================
    // INITIALIZATION
    // ============================================================================
    
    function initialize() {
        // Prevent multiple initializations
        if (window.icardAdminInitialized) {
            console.log('iCard admin already initialized, skipping...');
            return;
        }
        window.icardAdminInitialized = true;
        
        // Initialize test mode validation
        initializeTestModeValidation();
        
        // Add event listeners to test buttons
        var testButtons = document.querySelectorAll('#icard-test-btn-dev');
        testButtons.forEach(function(button) {
            // Check if listener already attached
            if (button.hasAttribute('data-icard-listener-attached')) {
                console.log('Event listener already attached to button, skipping...');
                return;
            }
            
            // Remove any existing listeners first
            button.removeEventListener('click', testIcardConnection);
            button.addEventListener('click', testIcardConnection);
            button.setAttribute('data-icard-listener-attached', 'true');
        });
        
        // Add CSRF token to forms if needed
        addCSRFToken();
        
        // Add production field validation
        addProductionFieldValidation();
    }
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initialize);
    } else {
        initialize();
    }
    
    // Make showToast globally available for external use
    window.showToast = showToast;
    
})();
